package ru.yandex.intranet.d.util.result

import reactor.core.publisher.Flux
import reactor.core.publisher.Mono
import reactor.util.function.Tuple2
import reactor.util.function.Tuples
import ru.yandex.intranet.d.datasource.model.WithTxId
import java.util.function.Consumer
import java.util.function.Function

/**
 * Result.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
sealed interface Result<T> {
    fun <R> match(cases: Cases<T, R>): R
    fun <R> match(onSuccess: Function<T, R>, onFailure: Function<ErrorCollection, R>): R
    fun <R> apply(map: Function<T, R>): Result<R>
    fun <R> applyMono(map: Function<T, Mono<R>>): Mono<Result<R>>
    fun <R> applyMonoTx(map: Function<T, Mono<WithTxId<R>>>): Mono<ResultTx<R>>
    fun <R> applyFlux(map: Function<T, Flux<R>>): Flux<Result<R>>
    fun <R> andThen(map: Function<T, Result<R>>): Result<R>
    fun <R> andThenMono(map: Function<T, Mono<Result<R>>>): Mono<Result<R>>
    fun <R> andThenFlux(map: Function<T, Flux<Result<R>>>): Flux<Result<R>>
    fun andThenDo(map: Function<T, Result<Void?>>): Result<T>
    fun doOnSuccess(consumer: Consumer<T>)
    fun doOnFailure(consumer: Consumer<ErrorCollection>)
    fun <R> cast(clazz: Class<R>): Result<R>
    fun toVoid(): Result<Void?>
    fun toUnit(): Result<Unit>
    fun isSuccess(): Boolean
    fun isFailure(): Boolean
    fun mapFailure(map: Function<ErrorCollection, ErrorCollection>): Result<T>
    suspend fun <R> matchSuspend(success: suspend (result: T) -> R, failure: suspend (error: ErrorCollection) -> R): R
    interface Cases<I, O> {
        fun success(result: I): O
        fun failure(error: ErrorCollection): O
    }
    companion object {
        @JvmStatic
        fun <I> success(result: I): Result<I> = Success(result)
        @JvmStatic
        fun <I> failure(error: ErrorCollection): Result<I> = Failure(error)
        @JvmStatic
        fun <U, V> zip(results: Tuple2<Result<U>, Result<V>>): Result<Tuple2<U, V>> {
            return results.t1.match({ resultLeft ->
                results.t2.match(
                    { resultRight -> Success(Tuples.of(resultLeft, resultRight)) },
                    { errorRight -> Failure(errorRight) }
                )
            }, { errorLeft ->
                results.t2.match(
                    { Failure(errorLeft) },
                    { errorRight -> Failure(ErrorCollection.builder().add(errorLeft).add(errorRight).build()) }
                )
            })
        }
    }
}
private data class Success<I>(val result: I): Result<I> {
    override fun <R> match(cases: Result.Cases<I, R>): R = cases.success(result)
    override fun <R> match(onSuccess: Function<I, R>, onFailure: Function<ErrorCollection, R>): R
            = onSuccess.apply(result)
    override fun <R> apply(map: Function<I, R>): Result<R> = Success(map.apply(result))
    override fun <R> applyMono(map: Function<I, Mono<R>>): Mono<Result<R>> = map.apply(result).map { Success(it) }
    override fun <R> applyMonoTx(map: Function<I, Mono<WithTxId<R>>>): Mono<ResultTx<R>>
            = map.apply(result).map { ResultTx.success(it.get(), it.transactionId) }
    override fun <R> applyFlux(map: Function<I, Flux<R>>): Flux<Result<R>> = map.apply(result).map { Success(it) }
    override fun <R> andThen(map: Function<I, Result<R>>): Result<R> = map.apply(result)
    override fun <R> andThenMono(map: Function<I, Mono<Result<R>>>): Mono<Result<R>> = map.apply(result)
    override fun <R> andThenFlux(map: Function<I, Flux<Result<R>>>): Flux<Result<R>> = map.apply(result)
    override fun andThenDo(map: Function<I, Result<Void?>>): Result<I>
            = map.apply(result).match({ this }, { Failure(it) })
    override fun doOnSuccess(consumer: Consumer<I>) = consumer.accept(result)
    override fun doOnFailure(consumer: Consumer<ErrorCollection>) {}
    override fun <R> cast(clazz: Class<R>): Result<R> = Success(clazz.cast(result))
    override fun toVoid(): Result<Void?> = Success(null)
    override fun toUnit(): Result<Unit> = Success(Unit)
    override fun isSuccess(): Boolean = true
    override fun isFailure(): Boolean = false
    override fun mapFailure(map: Function<ErrorCollection, ErrorCollection>): Result<I> = this
    override suspend fun <R> matchSuspend(success: suspend (result: I) -> R,
                                          failure: suspend (error: ErrorCollection) -> R): R = success(result)
}
private data class Failure<I>(val error: ErrorCollection): Result<I> {
    override fun <R> match(cases: Result.Cases<I, R>): R = cases.failure(error)
    override fun <R> match(onSuccess: Function<I, R>, onFailure: Function<ErrorCollection, R>): R
            = onFailure.apply(error)
    @Suppress("UNCHECKED_CAST")
    override fun <R> apply(map: Function<I, R>): Result<R> = this as Result<R>
    @Suppress("UNCHECKED_CAST")
    override fun <R> applyMono(map: Function<I, Mono<R>>): Mono<Result<R>> = Mono.just(this as Result<R>)
    override fun <R> applyMonoTx(map: Function<I, Mono<WithTxId<R>>>): Mono<ResultTx<R>>
            = Mono.just(ResultTx.failure(error, null))
    @Suppress("UNCHECKED_CAST")
    override fun <R> applyFlux(map: Function<I, Flux<R>>): Flux<Result<R>> = Flux.just(this as Result<R>)
    @Suppress("UNCHECKED_CAST")
    override fun <R> andThen(map: Function<I, Result<R>>): Result<R> = this as Result<R>
    @Suppress("UNCHECKED_CAST")
    override fun <R> andThenMono(map: Function<I, Mono<Result<R>>>): Mono<Result<R>> = Mono.just(this as Result<R>)
    @Suppress("UNCHECKED_CAST")
    override fun <R> andThenFlux(map: Function<I, Flux<Result<R>>>): Flux<Result<R>> = Flux.just(this as Result<R>)
    override fun andThenDo(map: Function<I, Result<Void?>>): Result<I> = this
    override fun doOnSuccess(consumer: Consumer<I>) {}
    override fun doOnFailure(consumer: Consumer<ErrorCollection>) = consumer.accept(error)
    @Suppress("UNCHECKED_CAST")
    override fun <R> cast(clazz: Class<R>): Result<R> = this as Result<R>
    override fun toVoid(): Result<Void?> = Failure(error)
    override fun toUnit(): Result<Unit> = Failure(error)
    override fun isSuccess(): Boolean = false
    override fun isFailure(): Boolean = true
    override fun mapFailure(map: Function<ErrorCollection, ErrorCollection>): Result<I> = Failure(map.apply(error))
    override suspend fun <R> matchSuspend(success: suspend (result: I) -> R,
                                          failure: suspend (error: ErrorCollection) -> R): R = failure(error)
}
