package ru.yandex.intranet.d.util.result

import reactor.core.publisher.Flux
import reactor.core.publisher.Mono
import reactor.util.function.Tuple2
import reactor.util.function.Tuples
import java.util.function.Consumer
import java.util.function.Function

/**
 * Result with transaction id.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
sealed interface ResultTx<T> {
    val transactionId: String?
    fun <R> match(cases: Cases<T, R>): R
    fun <R> match(onSuccess: Function<T, R>, onFailure: Function<ErrorCollection, R>): R
    fun <R> apply(map: Function<T, R>): ResultTx<R>
    fun <R> applyMono(map: Function<T, Mono<R>>): Mono<ResultTx<R>>
    fun <R> applyMono(onSuccess: Function<T, Mono<R>>,
                      onFailure: Function<ErrorCollection, Mono<ResultTx<R>>>): Mono<ResultTx<R>>
    fun <R> applyFlux(map: Function<T, Flux<R>>): Flux<ResultTx<R>>
    fun <R> andThen(map: Function<T, ResultTx<R>>): ResultTx<R>
    fun <R> andThenMono(map: Function<T, Mono<ResultTx<R>>>): Mono<ResultTx<R>>
    fun <R> andThenFlux(map: Function<T, Flux<ResultTx<R>>>): Flux<ResultTx<R>>
    fun andThenDo(map: Function<T, ResultTx<Void?>>): ResultTx<T>
    fun doOnSuccess(consumer: Consumer<T>)
    fun doOnFailure(consumer: Consumer<ErrorCollection>)
    fun <R> cast(clazz: Class<R>): ResultTx<R>
    fun toVoid(): ResultTx<Void?>
    fun toUnit(): ResultTx<Unit>
    fun isSuccess(): Boolean
    fun isFailure(): Boolean
    fun toResult(): Result<T>
    fun toResultClosed(): Result<T>
    suspend fun <R> matchSuspend(success: suspend (result: T) -> R, failure: suspend (error: ErrorCollection) -> R): R
    interface Cases<I, O> {
        fun success(result: I): O
        fun failure(error: ErrorCollection): O
    }
    companion object {
        @JvmStatic
        fun <I> success(result: I, transactionId: String?): ResultTx<I> = SuccessTx(result, transactionId)
        @JvmStatic
        fun <I> failure(error: ErrorCollection, transactionId: String?): ResultTx<I> = FailureTx(error, transactionId)
        @JvmStatic
        fun <U, V> zip(results: Tuple2<ResultTx<U>, ResultTx<V>>): ResultTx<Tuple2<U, V>> {
            val transactionId = results.t1.transactionId
            require(transactionId == results.t2.transactionId) { "Cannot zip results of different transactions." }
            return results.t1.match({ resultLeft ->
                results.t2.match(
                    { resultRight -> SuccessTx(Tuples.of(resultLeft, resultRight), transactionId) },
                    { errorRight -> FailureTx(errorRight, transactionId) }
                )
            }, { errorLeft ->
                results.t2.match(
                    { FailureTx(errorLeft, transactionId) },
                    { errorRight -> FailureTx(ErrorCollection.builder().add(errorLeft).add(errorRight).build(),
                        transactionId) }
                )
            })
        }
    }
}
private data class SuccessTx<I>(val result: I, override val transactionId: String?): ResultTx<I> {
    override fun <R> match(cases: ResultTx.Cases<I, R>): R = cases.success(result)
    override fun <R> match(onSuccess: Function<I, R>, onFailure: Function<ErrorCollection, R>): R
            = onSuccess.apply(result)
    override fun <R> apply(map: Function<I, R>): ResultTx<R> = SuccessTx(map.apply(result), transactionId)
    override fun <R> applyMono(map: Function<I, Mono<R>>): Mono<ResultTx<R>>
            = map.apply(result).map { SuccessTx(it, transactionId) }
    override fun <R> applyMono(onSuccess: Function<I, Mono<R>>,
                               onFailure: Function<ErrorCollection, Mono<ResultTx<R>>>): Mono<ResultTx<R>>
            = onSuccess.apply(result).map { SuccessTx(it, transactionId) }
    override fun <R> applyFlux(map: Function<I, Flux<R>>): Flux<ResultTx<R>>
            = map.apply(result).map { SuccessTx(it, transactionId) }
    override fun <R> andThen(map: Function<I, ResultTx<R>>): ResultTx<R> = map.apply(result)
    override fun <R> andThenMono(map: Function<I, Mono<ResultTx<R>>>): Mono<ResultTx<R>> = map.apply(result)
    override fun <R> andThenFlux(map: Function<I, Flux<ResultTx<R>>>): Flux<ResultTx<R>> = map.apply(result)
    override fun andThenDo(map: Function<I, ResultTx<Void?>>): ResultTx<I>
            = map.apply(result).match({ this }, { FailureTx(it, transactionId) })
    override fun doOnSuccess(consumer: Consumer<I>) = consumer.accept(result)
    override fun doOnFailure(consumer: Consumer<ErrorCollection>) {}
    override fun <R> cast(clazz: Class<R>): ResultTx<R> = SuccessTx(clazz.cast(result), transactionId)
    override fun toVoid(): ResultTx<Void?> = SuccessTx(null, transactionId)
    override fun toUnit(): ResultTx<Unit> = SuccessTx(Unit, transactionId)
    override fun isSuccess(): Boolean = true
    override fun isFailure(): Boolean = false
    override fun toResult(): Result<I> = Result.success(result)
    override fun toResultClosed(): Result<I> {
        check(transactionId == null || transactionId.isEmpty()) { "Transaction isn't closed!" }
        return Result.success(result)
    }
    override suspend fun <R> matchSuspend(success: suspend (result: I) -> R,
                                          failure: suspend (error: ErrorCollection) -> R): R = success(result)
}
private data class FailureTx<I>(val error: ErrorCollection, override val transactionId: String?): ResultTx<I> {
    override fun <R> match(cases: ResultTx.Cases<I, R>): R = cases.failure(error)
    override fun <R> match(onSuccess: Function<I, R>, onFailure: Function<ErrorCollection, R>): R
            = onFailure.apply(error)
    @Suppress("UNCHECKED_CAST")
    override fun <R> apply(map: Function<I, R>): ResultTx<R> = this as ResultTx<R>
    @Suppress("UNCHECKED_CAST")
    override fun <R> applyMono(map: Function<I, Mono<R>>): Mono<ResultTx<R>> = Mono.just(this as ResultTx<R>)
    override fun <R> applyMono(onSuccess: Function<I, Mono<R>>,
                               onFailure: Function<ErrorCollection, Mono<ResultTx<R>>>): Mono<ResultTx<R>>
            = onFailure.apply(error)
    @Suppress("UNCHECKED_CAST")
    override fun <R> applyFlux(map: Function<I, Flux<R>>): Flux<ResultTx<R>> = Flux.just(this as ResultTx<R>)
    @Suppress("UNCHECKED_CAST")
    override fun <R> andThen(map: Function<I, ResultTx<R>>): ResultTx<R> = this as ResultTx<R>
    @Suppress("UNCHECKED_CAST")
    override fun <R> andThenMono(map: Function<I, Mono<ResultTx<R>>>): Mono<ResultTx<R>>
            = Mono.just(this as ResultTx<R>)
    @Suppress("UNCHECKED_CAST")
    override fun <R> andThenFlux(map: Function<I, Flux<ResultTx<R>>>): Flux<ResultTx<R>>
            = Flux.just(this as ResultTx<R>)
    override fun andThenDo(map: Function<I, ResultTx<Void?>>): ResultTx<I> = this
    override fun doOnSuccess(consumer: Consumer<I>) {}
    override fun doOnFailure(consumer: Consumer<ErrorCollection>) = consumer.accept(error)
    @Suppress("UNCHECKED_CAST")
    override fun <R> cast(clazz: Class<R>): ResultTx<R> = this as ResultTx<R>
    override fun toVoid(): ResultTx<Void?> = FailureTx(error, transactionId)
    override fun toUnit(): ResultTx<Unit> = FailureTx(error, transactionId)
    override fun isSuccess(): Boolean = false
    override fun isFailure(): Boolean = true
    override fun toResult(): Result<I> = Result.failure(error)
    override fun toResultClosed(): Result<I> {
        check(transactionId == null || transactionId.isEmpty()) { "Transaction isn't closed!" }
        return Result.failure(error)
    }
    override suspend fun <R> matchSuspend(success: suspend (result: I) -> R,
                                          failure: suspend (error: ErrorCollection) -> R): R = failure(error)
}
