package ru.yandex.intranet.d.util.result;

import java.util.Locale;

import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;

import ru.yandex.intranet.d.services.quotas.QuotasHelper;
import ru.yandex.intranet.d.services.transfer.model.ValidatedQuotaResourceTransfer;
import ru.yandex.intranet.d.web.model.AmountDto;

@Component("transferRequestErrorMessages")
public class TransferRequestErrorMessages {
    private final MessageSource messages;

    public TransferRequestErrorMessages(@Qualifier("messageSource") MessageSource messages) {
        this.messages = messages;
    }

    public String getNotEnoughBalanceErrorMessage(
            ValidatedQuotaResourceTransfer resourceTransfer,
            long currentBalance,
            long currentProvided,
            Locale locale
    ) {
        AmountDto currentBalanceAmount = getAmountDto(currentBalance, resourceTransfer, locale);
        AmountDto currentProvidedAmount = getAmountDto(currentProvided, resourceTransfer, locale);
        return messages.getMessage("errors.not.enough.balance.for.quota.transfer", null, locale)
                + (currentProvided != 0 ? "\n" + messages
                .getMessage("errors.not.enough.balance.for.quota.transfer.non.zero.provided",
                        new Object[]{currentProvidedAmount.getReadableAmount() + " " +
                                currentProvidedAmount.getReadableUnit()}, locale) : "")
                + (currentBalance != 0 ? "\n" + messages
                .getMessage("errors.not.enough.balance.for.quota.transfer.non.zero.balance",
                        new Object[]{currentBalanceAmount.getRawAmount() + " " +
                                currentBalanceAmount.getRawUnit()}, locale) : "");
    }

    public String getNotEnoughAccountAvailableQuotaErrorMessage(ValidatedQuotaResourceTransfer resourceTransfer,
                                                                long currentProvided,
                                                                long currentAllocated,
                                                                long currentFrozen,
                                                                Locale locale) {
        return getAvailableAccountQuotaMessage(resourceTransfer, currentProvided, currentAllocated, currentFrozen,
                locale, "errors.not.enough.available.for.account.quota.transfer");
    }

    public String getProvidedQuotaLessThanRequestedErrorMessage(ValidatedQuotaResourceTransfer resourceTransfer,
                                                                long currentProvided,
                                                                long currentAllocated,
                                                                long currentFrozen,
                                                                Locale locale) {
        return getAvailableAccountQuotaMessage(resourceTransfer, currentProvided, currentAllocated, currentFrozen,
                locale, "errors.not.enough.provided.for.account.quota.transfer");
    }

    @NotNull
    private String getAvailableAccountQuotaMessage(
            ValidatedQuotaResourceTransfer resourceTransfer,
            long currentProvided,
            long currentAllocated,
            long currentFrozen,
            Locale locale,
            String code
    ) {
        long currentAvailable = Math.max(currentProvided - currentAllocated - currentFrozen, 0L);
        AmountDto currentProvidedAmount = getAmountDto(currentProvided, resourceTransfer, locale);
        AmountDto currentAvailableAmount = getAmountDto(currentAvailable, resourceTransfer, locale);
        AmountDto currentAllocatedAmount = getAmountDto(currentAllocated, resourceTransfer, locale);
        AmountDto currentFrozenAmount = getAmountDto(currentFrozen, resourceTransfer, locale);
        return messages.getMessage(code, null, locale)
                + ("\n" + getProvidedMessage(currentProvidedAmount, locale))
                + (currentAllocated != 0 ? "\n" + getAllocatedMessage(currentAllocatedAmount, locale) : "")
                + (currentFrozen != 0 ? "\n" + getFrozenMessage(currentFrozenAmount, locale) : "")
                + (currentAvailable != 0 ? "\n" + messages
                .getMessage("errors.not.enough.available.for.provision.transfer.non.zero.available",
                        new Object[]{currentAvailableAmount.getReadableAmount() + " " +
                                currentAvailableAmount.getReadableUnit()}, locale) : "");
    }

    @NotNull
    private AmountDto getAmountDto(long current, ValidatedQuotaResourceTransfer resourceTransfer,
                                   Locale locale) {
        return QuotasHelper.getAmountDto(QuotasHelper.toBigDecimal(current), resourceTransfer.getResource(),
                resourceTransfer.getResourceUnitsEnsemble(), locale);
    }

    public String getFrozenMessage(AmountDto frozen, Locale locale) {
        return getMessageWithAmount("errors.not.enough.available.for.provision.transfer.frozen.quota",
                frozen, locale);
    }

    public String getAllocatedMessage(AmountDto allocated, Locale locale) {
        return getMessageWithAmount("errors.not.enough.available.for.provision.transfer.allocated.quota",
                allocated, locale);
    }

    public String getProvidedMessage(AmountDto provided, Locale locale) {
        return getMessageWithAmount("errors.not.enough.available.for.provision.transfer.provided.quota",
                provided, locale);
    }

    private String getMessageWithAmount(String code, AmountDto amountDto, Locale locale) {
        return messages.getMessage(code, new Object[]{amountDto.getRawAmount() + " " + amountDto.getRawUnit()}, locale);
    }
}
