package ru.yandex.intranet.d.util.units;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.math.RoundingMode;
import java.text.NumberFormat;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuple3;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.units.GrammaticalCase;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.units.UnitsComparator;

/**
 * Units utils.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class Units {
    /**
     * It's used when it's necessary to obtain maximum precision and preserve all digits on units conversion.
     * */
    public static final MathContext MATH_CONTEXT = new MathContext(100, RoundingMode.HALF_EVEN);

    private Units() {
    }

    public static Tuple2<BigDecimal, UnitModel> convertToApi(long value, ResourceModel resource,
                                                             UnitsEnsembleModel unitsEnsemble) {
        return convertToApiInternal(value, resource, unitsEnsemble);
    }

    public static boolean canConvertToApi(long value, ResourceModel resource, UnitsEnsembleModel unitsEnsemble) {
        var result = convertToApiInternal(value, resource, unitsEnsemble);
        UnitModel unit = result.getT2();
        UnitModel preferredUnit = result.getT3();
        return unit.equals(preferredUnit);
    }

    public static Tuple2<BigDecimal, UnitModel> convertToApiStrictly(
            long value, ResourceModel resource, UnitsEnsembleModel unitsEnsemble
    ) {
        var result = convertToApiInternal(value, resource, unitsEnsemble);
        UnitModel unit = result.getT2();
        UnitModel preferredUnit = result.getT3();
        if (!unit.equals(preferredUnit)) {
            throw new IllegalStateException("Value of" + result.getT1() + " " + unit.getKey() +
                    "  cannot be converted to " + preferredUnit.getKey());
        }
        return result;
    }

    private static Tuple3<BigDecimal, UnitModel, UnitModel> convertToApiInternal(
            long value, ResourceModel resource, UnitsEnsembleModel unitsEnsemble
    ) {
        Optional<UnitModel> providerApiUnit = unitsEnsemble
                .unitById(resource.getResourceUnits().getProviderApiUnitId());
        Optional<UnitModel> minAllowedUnit = getMinAllowedUnit(resource, unitsEnsemble);
        UnitModel baseUnit = unitsEnsemble.unitById(resource.getBaseUnitId())
                .orElseThrow(() -> new IllegalStateException("Base unit not found " + resource.getBaseUnitId()));
        UnitModel preferredUnit = providerApiUnit.orElseGet(() -> minAllowedUnit.orElse(baseUnit));
        int unitsComparison = UnitsComparator.INSTANCE.compare(preferredUnit, baseUnit);
        if (preferredUnit.getId().equals(resource.getBaseUnitId()) || unitsComparison == 0) {
            return Tuples.of(BigDecimal.valueOf(value), preferredUnit, preferredUnit);
        }
        if (unitsComparison < 0) {
            throw new IllegalStateException("Invalid units configuration");
        }
        if (value == 0L) {
            return Tuples.of(BigDecimal.valueOf(value), preferredUnit, preferredUnit);
        }
        BigDecimal sign = value > 0 ? BigDecimal.ONE : BigDecimal.valueOf(-1L);
        BigDecimal absValue = BigDecimal.valueOf(Math.abs(value));
        if (preferredUnit.getBase() == baseUnit.getBase()) {
            long powerDiff = preferredUnit.getPower() - baseUnit.getPower();
            BigDecimal multiplier = BigDecimal.valueOf(preferredUnit.getBase()).pow((int) powerDiff);
            BigDecimal[] divided = absValue.divideAndRemainder(multiplier);
            BigDecimal reminder = divided[1];
            if (reminder.compareTo(BigDecimal.ZERO) == 0) {
                return Tuples.of(divided[0].multiply(sign), preferredUnit, preferredUnit);
            }
            return Tuples.of(BigDecimal.valueOf(value), baseUnit, preferredUnit);
        }
        if (baseUnit.getPower() >= 0 && preferredUnit.getPower() >= 0) {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase()).pow((int) baseUnit.getPower());
            BigDecimal preferredMultiplier = BigDecimal.valueOf(preferredUnit.getBase())
                    .pow((int) preferredUnit.getPower());
            BigDecimal[] divided = absValue.multiply(baseMultiplier)
                    .divideAndRemainder(preferredMultiplier);
            BigDecimal reminder = divided[1];
            if (reminder.compareTo(BigDecimal.ZERO) == 0) {
                return Tuples.of(divided[0].multiply(sign), preferredUnit, preferredUnit);
            }
            return Tuples.of(BigDecimal.valueOf(value), baseUnit, preferredUnit);
        } else if (baseUnit.getPower() < 0 && preferredUnit.getPower() >= 0) {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase())
                    .pow((int) (-1L * baseUnit.getPower()));
            BigDecimal preferredMultiplier = BigDecimal.valueOf(preferredUnit.getBase())
                    .pow((int) preferredUnit.getPower());
            BigDecimal[] divided = absValue
                    .divideAndRemainder(baseMultiplier.multiply(preferredMultiplier));
            BigDecimal reminder = divided[1];
            if (reminder.compareTo(BigDecimal.ZERO) == 0) {
                return Tuples.of(divided[0].multiply(sign), preferredUnit, preferredUnit);
            }
            return Tuples.of(BigDecimal.valueOf(value), baseUnit, preferredUnit);
        } else if (baseUnit.getPower() >= 0) {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase()).pow((int) baseUnit.getPower());
            BigDecimal preferredMultiplier = BigDecimal.valueOf(preferredUnit.getBase())
                    .pow((int) (-1L * preferredUnit.getPower()));
            BigDecimal converted = absValue.multiply(baseMultiplier).multiply(preferredMultiplier);
            if (converted.remainder(BigDecimal.ONE).compareTo(BigDecimal.ZERO) == 0) {
                return Tuples.of(converted.multiply(sign), preferredUnit, preferredUnit);
            }
            return Tuples.of(BigDecimal.valueOf(value), baseUnit, preferredUnit);
        } else {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase())
                    .pow((int) (-1L * baseUnit.getPower()));
            BigDecimal preferredMultiplier = BigDecimal.valueOf(preferredUnit.getBase())
                    .pow((int) (-1L * preferredUnit.getPower()));
            BigDecimal[] divided = absValue.multiply(preferredMultiplier)
                    .divideAndRemainder(baseMultiplier);
            BigDecimal reminder = divided[1];
            if (reminder.compareTo(BigDecimal.ZERO) == 0) {
                return Tuples.of(divided[0].multiply(sign), preferredUnit, preferredUnit);
            }
            return Tuples.of(BigDecimal.valueOf(value), baseUnit, preferredUnit);
        }
    }

    public static Tuple2<BigDecimal, UnitModel> convertToApi(BigInteger value, ResourceModel resource,
                                                             UnitsEnsembleModel unitsEnsemble) {
        Optional<UnitModel> providerApiUnit = unitsEnsemble
                .unitById(resource.getResourceUnits().getProviderApiUnitId());
        Optional<UnitModel> minAllowedUnit = getMinAllowedUnit(resource, unitsEnsemble);
        UnitModel baseUnit = unitsEnsemble.unitById(resource.getBaseUnitId())
                .orElseThrow(() -> new IllegalStateException("Base unit not found " + resource.getBaseUnitId()));
        UnitModel preferredUnit = providerApiUnit.orElseGet(() -> minAllowedUnit.orElse(baseUnit));
        if (value.compareTo(BigInteger.ZERO) == 0L) {
            return Tuples.of(new BigDecimal(value), preferredUnit);
        }
        int unitsComparison = UnitsComparator.INSTANCE.compare(preferredUnit, baseUnit);
        if (preferredUnit.getId().equals(resource.getBaseUnitId()) || unitsComparison == 0) {
            return Tuples.of(new BigDecimal(value), preferredUnit);
        }
        if (unitsComparison < 0) {
            throw new IllegalStateException("Invalid units configuration");
        }
        BigDecimal sign = value.compareTo(BigInteger.ZERO) > 0 ? BigDecimal.ONE : BigDecimal.valueOf(-1L);
        BigDecimal absValue = new BigDecimal(value.abs());
        if (preferredUnit.getBase() == baseUnit.getBase()) {
            long powerDiff = preferredUnit.getPower() - baseUnit.getPower();
            BigDecimal multiplier = BigDecimal.valueOf(preferredUnit.getBase()).pow((int) powerDiff);
            BigDecimal[] divided = absValue.divideAndRemainder(multiplier);
            BigDecimal reminder = divided[1];
            if (reminder.compareTo(BigDecimal.ZERO) == 0) {
                return Tuples.of(divided[0].multiply(sign), preferredUnit);
            }
            return Tuples.of(new BigDecimal(value), baseUnit);
        }
        if (baseUnit.getPower() >= 0 && preferredUnit.getPower() >= 0) {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase()).pow((int) baseUnit.getPower());
            BigDecimal preferredMultiplier = BigDecimal.valueOf(preferredUnit.getBase())
                    .pow((int) preferredUnit.getPower());
            BigDecimal[] divided = absValue.multiply(baseMultiplier)
                    .divideAndRemainder(preferredMultiplier);
            BigDecimal reminder = divided[1];
            if (reminder.compareTo(BigDecimal.ZERO) == 0) {
                return Tuples.of(divided[0].multiply(sign), preferredUnit);
            }
            return Tuples.of(new BigDecimal(value), baseUnit);
        } else if (baseUnit.getPower() < 0 && preferredUnit.getPower() >= 0) {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase())
                    .pow((int) (-1L * baseUnit.getPower()));
            BigDecimal preferredMultiplier = BigDecimal.valueOf(preferredUnit.getBase())
                    .pow((int) preferredUnit.getPower());
            BigDecimal[] divided = absValue
                    .divideAndRemainder(baseMultiplier.multiply(preferredMultiplier));
            BigDecimal reminder = divided[1];
            if (reminder.compareTo(BigDecimal.ZERO) == 0) {
                return Tuples.of(divided[0].multiply(sign), preferredUnit);
            }
            return Tuples.of(new BigDecimal(value), baseUnit);
        } else if (baseUnit.getPower() >= 0) {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase()).pow((int) baseUnit.getPower());
            BigDecimal preferredMultiplier = BigDecimal.valueOf(preferredUnit.getBase())
                    .pow((int) (-1L * preferredUnit.getPower()));
            BigDecimal converted = absValue.multiply(baseMultiplier).multiply(preferredMultiplier);
            if (converted.remainder(BigDecimal.ONE).compareTo(BigDecimal.ZERO) == 0) {
                return Tuples.of(converted.multiply(sign), preferredUnit);
            }
            return Tuples.of(new BigDecimal(value), baseUnit);
        } else {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase())
                    .pow((int) (-1L * baseUnit.getPower()));
            BigDecimal preferredMultiplier = BigDecimal.valueOf(preferredUnit.getBase())
                    .pow((int) (-1L * preferredUnit.getPower()));
            BigDecimal[] divided = absValue.multiply(preferredMultiplier)
                    .divideAndRemainder(baseMultiplier);
            BigDecimal reminder = divided[1];
            if (reminder.compareTo(BigDecimal.ZERO) == 0) {
                return Tuples.of(divided[0].multiply(sign), preferredUnit);
            }
            return Tuples.of(new BigDecimal(value), baseUnit);
        }
    }

    public static Optional<UnitModel> getMinAllowedUnit(ResourceModel resource, UnitsEnsembleModel unitsEnsemble) {
        Set<String> allowedUnitIds = resource.getResourceUnits().getAllowedUnitIds();
        return unitsEnsemble.getUnits().stream()
                .filter(u -> allowedUnitIds.contains(u.getId()))
                .min(UnitsComparator.INSTANCE);
    }

    public static Optional<UnitModel> getMinAllowedUnit(Set<String> allowedUnitIds, UnitsEnsembleModel unitsEnsemble) {
        return unitsEnsemble.getUnits().stream()
                .filter(u -> allowedUnitIds.contains(u.getId()))
                .min(UnitsComparator.INSTANCE);
    }

    public static Optional<Long> convertFromApi(long apiValue, ResourceModel resource,
                                                UnitsEnsembleModel ensemble, UnitModel unit) {
        if (resource.getBaseUnitId().equals(unit.getId())) {
            return Optional.of(apiValue);
        }
        UnitModel baseUnit = ensemble.unitById(resource.getBaseUnitId())
                .orElseThrow(() -> new IllegalArgumentException("Invalid units ensemble"));
        if (ensemble.unitById(unit.getId()).isEmpty()) {
            throw new IllegalArgumentException("Invalid units ensemble");
        }
        int unitsComparison = UnitsComparator.INSTANCE.compare(unit, baseUnit);
        if (unitsComparison == 0) {
            return Optional.of(apiValue);
        }
        if (unitsComparison < 0) {
            throw new IllegalStateException("Invalid units configuration");
        }
        if (baseUnit.getBase() == unit.getBase()) {
            long powerDiff = unit.getPower() - baseUnit.getPower();
            BigDecimal multiplier = BigDecimal.valueOf(unit.getBase()).pow((int) powerDiff);
            BigDecimal converted = BigDecimal.valueOf(apiValue).multiply(multiplier);
            if (converted.compareTo(BigDecimal.valueOf(Long.MAX_VALUE)) > 0
                    || converted.compareTo(BigDecimal.valueOf(Long.MIN_VALUE)) < 0) {
                return Optional.empty();
            }
            return Optional.of(converted.longValue());
        }
        BigDecimal sign = apiValue > 0 ? BigDecimal.ONE : BigDecimal.valueOf(-1L);
        BigDecimal absValue = BigDecimal.valueOf(Math.abs(apiValue));
        if (baseUnit.getPower() >= 0 && unit.getPower() >= 0) {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase()).pow((int) baseUnit.getPower());
            BigDecimal multiplier = BigDecimal.valueOf(unit.getBase()).pow((int) unit.getPower());
            BigDecimal[] divided = absValue.multiply(multiplier).divideAndRemainder(baseMultiplier);
            BigDecimal reminder = divided[1];
            if (reminder.compareTo(BigDecimal.ZERO) == 0) {
                BigDecimal converted = divided[0].multiply(sign);
                if (converted.compareTo(BigDecimal.valueOf(Long.MAX_VALUE)) > 0
                        || converted.compareTo(BigDecimal.valueOf(Long.MIN_VALUE)) < 0) {
                    return Optional.empty();
                }
                return Optional.of(converted.longValue());
            }
            return Optional.empty();
        } else if (baseUnit.getPower() < 0 && unit.getPower() >= 0) {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase()).pow((int) (-1L * baseUnit.getPower()));
            BigDecimal multiplier = BigDecimal.valueOf(unit.getBase()).pow((int) unit.getPower());
            BigDecimal converted = BigDecimal.valueOf(apiValue).multiply(multiplier).multiply(baseMultiplier);
            if (converted.compareTo(BigDecimal.valueOf(Long.MAX_VALUE)) > 0
                    || converted.compareTo(BigDecimal.valueOf(Long.MIN_VALUE)) < 0) {
                return Optional.empty();
            }
            return Optional.of(converted.longValue());
        } else if (baseUnit.getPower() >= 0 && unit.getPower() < 0) {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase()).pow((int) baseUnit.getPower());
            BigDecimal multiplier = BigDecimal.valueOf(unit.getBase()).pow((int) (-1L * unit.getPower()));
            BigDecimal[] divided = absValue.divideAndRemainder(multiplier.multiply(baseMultiplier));
            BigDecimal reminder = divided[1];
            if (reminder.compareTo(BigDecimal.ZERO) == 0) {
                BigDecimal converted = divided[0].multiply(sign);
                if (converted.compareTo(BigDecimal.valueOf(Long.MAX_VALUE)) > 0
                        || converted.compareTo(BigDecimal.valueOf(Long.MIN_VALUE)) < 0) {
                    return Optional.empty();
                }
                return Optional.of(converted.longValue());
            }
            return Optional.empty();
        } else {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase()).pow((int) (-1L * baseUnit.getPower()));
            BigDecimal multiplier = BigDecimal.valueOf(unit.getBase()).pow((int) (-1L * unit.getPower()));
            BigDecimal[] divided = absValue.multiply(baseMultiplier).divideAndRemainder(multiplier);
            BigDecimal reminder = divided[1];
            if (reminder.compareTo(BigDecimal.ZERO) == 0) {
                BigDecimal converted = divided[0].multiply(sign);
                if (converted.compareTo(BigDecimal.valueOf(Long.MAX_VALUE)) > 0
                        || converted.compareTo(BigDecimal.valueOf(Long.MIN_VALUE)) < 0) {
                    return Optional.empty();
                }
                return Optional.of(converted.longValue());
            }
            return Optional.empty();
        }
    }

    public static Optional<Long> subtract(long left, long right) {
        BigInteger difference = BigInteger.valueOf(left).subtract(BigInteger.valueOf(right));
        if (difference.compareTo(BigInteger.valueOf(Long.MAX_VALUE)) > 0
                || difference.compareTo(BigInteger.valueOf(Long.MIN_VALUE)) < 0) {
            return Optional.empty();
        }
        return Optional.of(difference.longValue());
    }

    public static Optional<Long> add(long left, long right) {
        BigInteger sum = BigInteger.valueOf(left).add(BigInteger.valueOf(right));
        if (sum.compareTo(BigInteger.valueOf(Long.MAX_VALUE)) > 0
                || sum.compareTo(BigInteger.valueOf(Long.MIN_VALUE)) < 0) {
            return Optional.empty();
        }
        return Optional.of(sum.longValue());
    }

    public static OptionalLong longValue(BigInteger value) {
        if (value.compareTo(BigInteger.valueOf(Long.MAX_VALUE)) > 0
                || value.compareTo(BigInteger.valueOf(Long.MIN_VALUE)) < 0) {
            return OptionalLong.empty();
        }
        return OptionalLong.of(value.longValue());
    }

    public static BigDecimal convert(BigDecimal amount, UnitModel fromUnit, UnitModel toUnit) {
        if (amount == null) {
            return BigDecimal.ZERO;
        }
        if (toUnit.getBase() == fromUnit.getBase() && toUnit.getPower() == fromUnit.getPower()) {
            return amount;
        }

        int fromUnitPower = Math.toIntExact(fromUnit.getPower());
        BigDecimal fromUnitBase = BigDecimal.valueOf(fromUnit.getBase());
        int toUnitPower = Math.toIntExact(toUnit.getPower());
        BigDecimal toUnitBase = BigDecimal.valueOf(toUnit.getBase());

        // (fromUnitBase ^ fromUnitBase) / (toUnitBase ^ toUnitPower)  as a rational number (ordinary fraction)
        BigDecimal numerator;
        BigDecimal denominator;
        if (fromUnit.getBase() == toUnit.getBase()) {
            //noinspection UnnecessaryLocalVariable
            BigDecimal base = fromUnitBase;
            int power = fromUnitPower - toUnitPower;
            if (power == 0) {
                numerator = BigDecimal.ONE;
                denominator = BigDecimal.ONE;
            } else if (power > 0) {
                numerator = base.pow(power);
                denominator = BigDecimal.ONE;
            } else {
                numerator = BigDecimal.ONE;
                denominator = base.pow(Math.abs(power));
            }
        } else {
            numerator = fromUnitPower == 0
                    ? BigDecimal.ONE
                    : (fromUnitPower > 0
                        ? fromUnitBase.pow(fromUnitPower)
                        : BigDecimal.ONE.divide(fromUnitBase.pow(Math.abs(fromUnitPower)), MATH_CONTEXT));
            denominator = toUnitPower == 0
                    ? BigDecimal.ONE
                    : (toUnitPower > 0
                        ? toUnitBase.pow(toUnitPower)
                        : BigDecimal.ONE.divide(toUnitBase.pow(Math.abs(toUnitPower)), MATH_CONTEXT));
        }

        return amount.multiply(numerator).divide(denominator, MATH_CONTEXT);
    }

    public static Tuple2<BigDecimal, UnitModel> convertToQuotaTransferRequestDisplay(
            long value, ResourceModel resource, UnitsEnsembleModel unitsEnsemble) {
        List<UnitModel> allUnits = unitsEnsemble.getUnits().stream()
                .sorted(UnitsComparator.INSTANCE).collect(Collectors.toList());
        Set<String> allowedUnitIds =  resource.getResourceUnits().getAllowedUnitIds();
        List<UnitModel> allowedUnits = allUnits.stream()
                .filter(u -> allowedUnitIds.contains(u.getId())).collect(Collectors.toList());
        UnitModel baseUnit = unitsEnsemble.unitById(resource.getBaseUnitId())
                .orElseThrow(() -> new IllegalStateException("Base unit not found " + resource.getBaseUnitId()));
        UnitModel minAllowedUnit = allowedUnits.stream().findFirst().orElse(baseUnit);
        UnitModel maxAllowedUnit = !allowedUnits.isEmpty() ? allowedUnits.get(allowedUnits.size() - 1) : baseUnit;
        if (value == 0L) {
            return Tuples.of(BigDecimal.ZERO, minAllowedUnit);
        }
        BigDecimal sign = value > 0 ? BigDecimal.ONE : BigDecimal.valueOf(-1L);
        BigDecimal absValue = BigDecimal.valueOf(Math.abs(value));
        for (UnitModel unit : Lists.reverse(allUnits)) {
            int compareToMaxAllowed = UnitsComparator.INSTANCE.compare(unit, maxAllowedUnit);
            if (compareToMaxAllowed > 0) {
                continue;
            }
            int compareToBase = UnitsComparator.INSTANCE.compare(unit, baseUnit);
            if (compareToBase == 0) {
                return Tuples.of(BigDecimal.valueOf(value), baseUnit);
            }
            if (compareToBase < 0) {
                throw new IllegalStateException("Invalid units configuration");
            }
            if (unit.getBase() == baseUnit.getBase()) {
                long powerDiff = unit.getPower() - baseUnit.getPower();
                BigDecimal multiplier = BigDecimal.valueOf(unit.getBase()).pow((int) powerDiff);
                BigDecimal[] divided = absValue.divideAndRemainder(multiplier);
                BigDecimal quotient = divided[0];
                BigDecimal reminder = divided[1];
                if (reminder.compareTo(BigDecimal.ZERO) == 0 && quotient.compareTo(BigDecimal.ONE) >= 0) {
                    return Tuples.of(quotient.multiply(sign), unit);
                }
            }
            if (baseUnit.getPower() >= 0 && unit.getPower() >= 0) {
                BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase()).pow((int) baseUnit.getPower());
                BigDecimal unitMultiplier = BigDecimal.valueOf(unit.getBase())
                        .pow((int) unit.getPower());
                BigDecimal[] divided = absValue.multiply(baseMultiplier)
                        .divideAndRemainder(unitMultiplier);
                BigDecimal quotient = divided[0];
                BigDecimal reminder = divided[1];
                if (reminder.compareTo(BigDecimal.ZERO) == 0 && quotient.compareTo(BigDecimal.ONE) >= 0) {
                    return Tuples.of(quotient.multiply(sign), unit);
                }
            } else if (baseUnit.getPower() < 0 && unit.getPower() >= 0) {
                BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase())
                        .pow((int) (-1L * baseUnit.getPower()));
                BigDecimal unitMultiplier = BigDecimal.valueOf(unit.getBase())
                        .pow((int) unit.getPower());
                BigDecimal[] divided = absValue
                        .divideAndRemainder(baseMultiplier.multiply(unitMultiplier));
                BigDecimal quotient = divided[0];
                BigDecimal reminder = divided[1];
                if (reminder.compareTo(BigDecimal.ZERO) == 0 && quotient.compareTo(BigDecimal.ONE) >= 0) {
                    return Tuples.of(quotient.multiply(sign), unit);
                }
            } else if (baseUnit.getPower() >= 0) {
                BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase()).pow((int) baseUnit.getPower());
                BigDecimal unitMultiplier = BigDecimal.valueOf(unit.getBase())
                        .pow((int) (-1L * unit.getPower()));
                BigDecimal converted = absValue.multiply(baseMultiplier).multiply(unitMultiplier);
                if (converted.remainder(BigDecimal.ONE).compareTo(BigDecimal.ZERO) == 0
                        && converted.compareTo(BigDecimal.ONE) >= 0) {
                    return Tuples.of(converted.multiply(sign), unit);
                }
            } else {
                BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase())
                        .pow((int) (-1L * baseUnit.getPower()));
                BigDecimal unitMultiplier = BigDecimal.valueOf(unit.getBase())
                        .pow((int) (-1L * unit.getPower()));
                BigDecimal[] divided = absValue.multiply(unitMultiplier)
                        .divideAndRemainder(baseMultiplier);
                BigDecimal quotient = divided[0];
                BigDecimal reminder = divided[1];
                if (reminder.compareTo(BigDecimal.ZERO) == 0 && quotient.compareTo(BigDecimal.ONE) >= 0) {
                    return Tuples.of(quotient.multiply(sign), unit);
                }
            }
        }
        return Tuples.of(BigDecimal.valueOf(value), baseUnit);
    }

    public static String formatQuotaTransferRequestAmount(BigDecimal value, Locale locale) {
        NumberFormat numberFormat = NumberFormat.getIntegerInstance(locale);
        numberFormat.setGroupingUsed(false);
        return numberFormat.format(value);
    }

    public static BigDecimal convertFromUnitToUnitRoundDownToInteger(BigDecimal amount,
                                                                     UnitsEnsembleModel ensemble,
                                                                     UnitModel fromUnit,
                                                                     UnitModel toUnit) {
        if (ensemble.unitById(fromUnit.getId()).isEmpty() || ensemble.unitById(toUnit.getId()).isEmpty()) {
            throw new IllegalArgumentException("Invalid units ensemble");
        }
        int unitsComparison = UnitsComparator.INSTANCE.compare(fromUnit, toUnit);
        if (unitsComparison == 0) {
            return roundDownToInteger(amount);
        }
        if (toUnit.getBase() == fromUnit.getBase()) {
            if (fromUnit.getPower() >= toUnit.getPower()) {
                long powerDiff = fromUnit.getPower() - toUnit.getPower();
                BigDecimal multiplier = BigDecimal.valueOf(fromUnit.getBase()).pow((int) powerDiff);
                BigDecimal converted = amount.multiply(multiplier);
                return roundDownToInteger(converted);
            } else {
                long powerDiff = toUnit.getPower() - fromUnit.getPower();
                BigDecimal divisor = BigDecimal.valueOf(fromUnit.getBase()).pow((int) powerDiff);
                BigDecimal converted = amount.divide(divisor, 0, RoundingMode.DOWN);
                return roundDownToInteger(converted);
            }
        }
        if (toUnit.getPower() >= 0 && fromUnit.getPower() >= 0) {
            BigDecimal toMultiplier = BigDecimal.valueOf(toUnit.getBase()).pow((int) toUnit.getPower());
            BigDecimal fromMultiplier = BigDecimal.valueOf(fromUnit.getBase()).pow((int) fromUnit.getPower());
            BigDecimal converted = amount.multiply(fromMultiplier).divide(toMultiplier, 0, RoundingMode.DOWN);
            return roundDownToInteger(converted);
        } else if (toUnit.getPower() < 0 && fromUnit.getPower() >= 0) {
            BigDecimal toMultiplier = BigDecimal.valueOf(toUnit.getBase()).pow((int) (-1L * toUnit.getPower()));
            BigDecimal fromMultiplier = BigDecimal.valueOf(fromUnit.getBase()).pow((int) fromUnit.getPower());
            BigDecimal converted = amount.multiply(fromMultiplier).multiply(toMultiplier);
            return roundDownToInteger(converted);
        } else if (toUnit.getPower() >= 0 && fromUnit.getPower() < 0) {
            BigDecimal toMultiplier = BigDecimal.valueOf(toUnit.getBase()).pow((int) toUnit.getPower());
            BigDecimal fromMultiplier = BigDecimal.valueOf(fromUnit.getBase()).pow((int) (-1L * fromUnit.getPower()));
            BigDecimal converted = amount.divide(fromMultiplier.multiply(toMultiplier), 0, RoundingMode.DOWN);
            return roundDownToInteger(converted);
        } else {
            BigDecimal toMultiplier = BigDecimal.valueOf(toUnit.getBase()).pow((int) (-1L * toUnit.getPower()));
            BigDecimal fromMultiplier = BigDecimal.valueOf(fromUnit.getBase()).pow((int) (-1L * fromUnit.getPower()));
            BigDecimal converted = amount.multiply(toMultiplier).divide(fromMultiplier, 0, RoundingMode.DOWN);
            return roundDownToInteger(converted);
        }
    }

    public static Optional<Long> convertToBaseUnit(BigDecimal amount, ResourceModel resource,
                                                   UnitsEnsembleModel ensemble, UnitModel unit) {
        UnitModel baseUnit = ensemble.unitById(resource.getBaseUnitId())
                .orElseThrow(() -> new IllegalArgumentException("Invalid units ensemble"));
        if (ensemble.unitById(unit.getId()).isEmpty()) {
            throw new IllegalArgumentException("Invalid units ensemble");
        }
        int unitsComparison = UnitsComparator.INSTANCE.compare(unit, baseUnit);
        if (unitsComparison == 0) {
            if (amount.compareTo(BigDecimal.valueOf(Long.MAX_VALUE)) > 0
                    || amount.compareTo(BigDecimal.valueOf(Long.MIN_VALUE)) < 0) {
                return Optional.empty();
            }
            if (amount.abs().remainder(BigDecimal.ONE).compareTo(BigDecimal.ZERO) != 0) {
                return Optional.empty();
            }
            return Optional.of(amount.longValue());
        }
        if (unitsComparison < 0) {
            throw new IllegalStateException("Invalid units configuration");
        }
        if (baseUnit.getBase() == unit.getBase()) {
            long powerDiff = unit.getPower() - baseUnit.getPower();
            BigDecimal multiplier = BigDecimal.valueOf(unit.getBase()).pow((int) powerDiff);
            BigDecimal converted = amount.multiply(multiplier);
            if (converted.compareTo(BigDecimal.valueOf(Long.MAX_VALUE)) > 0
                    || converted.compareTo(BigDecimal.valueOf(Long.MIN_VALUE)) < 0) {
                return Optional.empty();
            }
            if (converted.abs().remainder(BigDecimal.ONE).compareTo(BigDecimal.ZERO) != 0) {
                return Optional.empty();
            }
            return Optional.of(converted.longValue());
        }
        BigDecimal sign = amount.compareTo(BigDecimal.ZERO) > 0 ? BigDecimal.ONE : BigDecimal.valueOf(-1L);
        BigDecimal absValue = amount.abs();
        if (baseUnit.getPower() >= 0 && unit.getPower() >= 0) {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase()).pow((int) baseUnit.getPower());
            BigDecimal multiplier = BigDecimal.valueOf(unit.getBase()).pow((int) unit.getPower());
            BigDecimal[] divided = absValue.multiply(multiplier).divideAndRemainder(baseMultiplier);
            BigDecimal reminder = divided[1];
            if (reminder.compareTo(BigDecimal.ZERO) == 0) {
                BigDecimal converted = divided[0].multiply(sign);
                if (converted.compareTo(BigDecimal.valueOf(Long.MAX_VALUE)) > 0
                        || converted.compareTo(BigDecimal.valueOf(Long.MIN_VALUE)) < 0) {
                    return Optional.empty();
                }
                return Optional.of(converted.longValue());
            }
            return Optional.empty();
        } else if (baseUnit.getPower() < 0 && unit.getPower() >= 0) {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase()).pow((int) (-1L * baseUnit.getPower()));
            BigDecimal multiplier = BigDecimal.valueOf(unit.getBase()).pow((int) unit.getPower());
            BigDecimal converted = amount.multiply(multiplier).multiply(baseMultiplier);
            if (converted.compareTo(BigDecimal.valueOf(Long.MAX_VALUE)) > 0
                    || converted.compareTo(BigDecimal.valueOf(Long.MIN_VALUE)) < 0) {
                return Optional.empty();
            }
            if (converted.abs().remainder(BigDecimal.ONE).compareTo(BigDecimal.ZERO) != 0) {
                return Optional.empty();
            }
            return Optional.of(converted.longValue());
        } else if (baseUnit.getPower() >= 0 && unit.getPower() < 0) {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase()).pow((int) baseUnit.getPower());
            BigDecimal multiplier = BigDecimal.valueOf(unit.getBase()).pow((int) (-1L * unit.getPower()));
            BigDecimal[] divided = absValue.divideAndRemainder(multiplier.multiply(baseMultiplier));
            BigDecimal reminder = divided[1];
            if (reminder.compareTo(BigDecimal.ZERO) == 0) {
                BigDecimal converted = divided[0].multiply(sign);
                if (converted.compareTo(BigDecimal.valueOf(Long.MAX_VALUE)) > 0
                        || converted.compareTo(BigDecimal.valueOf(Long.MIN_VALUE)) < 0) {
                    return Optional.empty();
                }
                return Optional.of(converted.longValue());
            }
            return Optional.empty();
        } else {
            BigDecimal baseMultiplier = BigDecimal.valueOf(baseUnit.getBase()).pow((int) (-1L * baseUnit.getPower()));
            BigDecimal multiplier = BigDecimal.valueOf(unit.getBase()).pow((int) (-1L * unit.getPower()));
            BigDecimal[] divided = absValue.multiply(baseMultiplier).divideAndRemainder(multiplier);
            BigDecimal reminder = divided[1];
            if (reminder.compareTo(BigDecimal.ZERO) == 0) {
                BigDecimal converted = divided[0].multiply(sign);
                if (converted.compareTo(BigDecimal.valueOf(Long.MAX_VALUE)) > 0
                        || converted.compareTo(BigDecimal.valueOf(Long.MIN_VALUE)) < 0) {
                    return Optional.empty();
                }
                return Optional.of(converted.longValue());
            }
            return Optional.empty();
        }
    }

    public static String toShortUnitName(BigDecimal amount, UnitModel unit, Locale locale) {
        if (locale.equals(Locales.RUSSIAN)) {
            boolean fractional = amount.abs().remainder(BigDecimal.ONE).compareTo(BigDecimal.ZERO) != 0;
            if (fractional) {
                return unit.getShortNameSingularRu().get(GrammaticalCase.GENITIVE);
            }
            BigDecimal value = amount.remainder(BigDecimal.valueOf(100));
            BigDecimal num = value.remainder(BigDecimal.TEN);
            if (value.compareTo(BigDecimal.TEN) > 0 && value.compareTo(BigDecimal.valueOf(20)) < 0) {
                return unit.getShortNamePluralRu().get(GrammaticalCase.GENITIVE);
            }
            if (num.compareTo(BigDecimal.ONE) > 0 && num.compareTo(BigDecimal.valueOf(5)) < 0) {
                return unit.getShortNameSingularRu().get(GrammaticalCase.GENITIVE);
            }
            if (num.compareTo(BigDecimal.ONE) == 0) {
                return unit.getShortNameSingularRu().get(GrammaticalCase.NOMINATIVE);
            }
            return unit.getShortNamePluralRu().get(GrammaticalCase.GENITIVE);
        }
        return amount.compareTo(BigDecimal.ONE) == 0 ? unit.getShortNameSingularEn() : unit.getShortNamePluralEn();
    }

    private static BigDecimal roundDownToInteger(BigDecimal amount) {
        if (amount.abs().remainder(BigDecimal.ONE).compareTo(BigDecimal.ZERO) != 0) {
            return amount.setScale(0, RoundingMode.DOWN);
        }
        return amount;
    }

}
