package ru.yandex.intranet.d.web;

import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.web.WebProperties;
import org.springframework.boot.web.codec.CodecCustomizer;
import org.springframework.boot.web.reactive.error.ErrorAttributes;
import org.springframework.boot.web.reactive.error.ErrorWebExceptionHandler;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.i18n.LocaleContext;
import org.springframework.context.i18n.SimpleLocaleContext;
import org.springframework.core.annotation.Order;
import org.springframework.http.MediaType;
import org.springframework.http.codec.ServerCodecConfigurer;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.lang.NonNull;
import org.springframework.util.StringUtils;
import org.springframework.web.reactive.config.DelegatingWebFluxConfiguration;
import org.springframework.web.reactive.result.view.ViewResolver;
import org.springframework.web.server.ServerWebExchange;
import org.springframework.web.server.i18n.AcceptHeaderLocaleContextResolver;
import org.springframework.web.server.i18n.LocaleContextResolver;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.web.errors.YaErrorWebExceptionHandler;
import ru.yandex.intranet.d.web.jackson.CustomJackson2Decoder;
import ru.yandex.intranet.d.web.log.AccessLogAttributesProducer;

import static ru.yandex.intranet.d.i18n.Locales.SUPPORTED_LOCALES;

/**
 * Webflux configuration.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Configuration
public class WebFluxConfiguration extends DelegatingWebFluxConfiguration {

    private final ObjectProvider<CodecCustomizer> codecCustomizers;

    public WebFluxConfiguration(ObjectProvider<CodecCustomizer> codecCustomizers) {
        this.codecCustomizers = codecCustomizers;
    }

    @NonNull
    @Override
    public LocaleContextResolver createLocaleContextResolver() {
        AcceptHeaderLocaleContextResolver localeContextResolver = new AcceptHeaderLocaleContextResolver() {
            @Override
            @NonNull
            public LocaleContext resolveLocaleContext(ServerWebExchange exchange) {
                ServerHttpRequest request = exchange.getRequest();
                List<String> lang = request.getQueryParams().get("lang");
                if (lang != null && lang.size() == 1) {
                    Locale locale = StringUtils.parseLocaleString(lang.get(0));
                    if (locale != null && SUPPORTED_LOCALES.contains(locale)) {
                        return new SimpleLocaleContext(locale);
                    }
                }
                return super.resolveLocaleContext(exchange);
            }
        };
        localeContextResolver.setDefaultLocale(Locales.ENGLISH);
        localeContextResolver.setSupportedLocales(SUPPORTED_LOCALES);
        return localeContextResolver;
    }

    @Bean
    @Order(-2)
    public ErrorWebExceptionHandler errorWebExceptionHandler(ErrorAttributes errorAttributes,
                                                             WebProperties webProperties,
                                                             ObjectProvider<ViewResolver> viewResolvers,
                                                             ServerCodecConfigurer serverCodecConfigurer,
                                                             ApplicationContext applicationContext,
                                                             AccessLogAttributesProducer accessLogAttributesProducer) {
        YaErrorWebExceptionHandler exceptionHandler = new YaErrorWebExceptionHandler(errorAttributes,
                webProperties.getResources(), applicationContext, accessLogAttributesProducer);
        exceptionHandler.setViewResolvers(viewResolvers.orderedStream().collect(Collectors.toList()));
        exceptionHandler.setMessageWriters(serverCodecConfigurer.getWriters());
        exceptionHandler.setMessageReaders(serverCodecConfigurer.getReaders());
        return exceptionHandler;
    }

    @Override
    protected void configureHttpMessageCodecs(@NonNull ServerCodecConfigurer configurer) {
        this.codecCustomizers.orderedStream().forEach((customizer) -> customizer.customize(configurer));
    }

    @Configuration
    public static class Jackson2Configuration {
        @Bean
        @Order(2)
        public CodecCustomizer jackson2Configuration(ObjectMapper objectMapper) {
            return configurer -> configurer.defaultCodecs()
                    .jackson2JsonDecoder(new CustomJackson2Decoder(objectMapper, MediaType.APPLICATION_JSON));
        }
    }
}
