package ru.yandex.intranet.d.web.controllers;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UncheckedIOException;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Schedulers;

import ru.yandex.intranet.d.web.security.roles.SolomonFetcherRole;
import ru.yandex.monlib.metrics.encode.spack.MetricSpackEncoder;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * Metrics endpoints handlers.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@RestController
@RequestMapping("/sensors")
@SolomonFetcherRole
public class SensorsController {

    private static final String SPACK_MIME_TYPE = "application/x-solomon-spack";

    @Operation(description = "Return spack metrics.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested metrics.",
            content = @Content(mediaType = SPACK_MIME_TYPE))})
    @GetMapping(path = "/metrics", produces = SPACK_MIME_TYPE)
    public Mono<ResponseEntity<byte[]>> getMetrics() {
        return Mono.empty().publishOn(Schedulers.boundedElastic())
                .then(Mono.defer(() -> Mono.just(ResponseEntity.ok()
                        .contentType(MediaType.valueOf(SPACK_MIME_TYPE))
                        .body(getEncodedMetrics()))));
    }

    private byte[] getEncodedMetrics() {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream()) {
            try (MetricSpackEncoder encoder = new MetricSpackEncoder(TimePrecision.SECONDS,
                    CompressionAlg.ZSTD, baos)) {
                MetricRegistry.root().supply(0, encoder);
            }
            return baos.toByteArray();
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

}
