package ru.yandex.intranet.d.web.controllers.admin.coordination;

import java.security.Principal;
import java.util.Locale;
import java.util.Set;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.coordination.model.cluster.NodeInfo;
import ru.yandex.intranet.d.datasource.coordination.spring.AutoClusterManager;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.coordination.ClusterNodeDto;
import ru.yandex.intranet.d.web.model.coordination.ClusterStateDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserRole;

/**
 * Coordination admin controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserRole
@RestController
@RequestMapping("/admin/coordination")
public class CoordinationController {

    private final MessageSource messages;
    private final AutoClusterManager clusterManager;

    public CoordinationController(@Qualifier("messageSource") MessageSource messages,
                                  AutoClusterManager clusterManager) {
        this.messages = messages;
        this.clusterManager = clusterManager;
    }

    @Operation(summary = "Get cluster state.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Cluster state.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ClusterStateDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getClusterState(Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        return Mono.just(checkAdminPermissions(currentUser, locale).apply(v -> getCachedClusterState())
                .match(Responses::okJson, Errors::toResponse));
    }

    private Result<Void> checkAdminPermissions(YaUserDetails currentUser, Locale locale) {
        if (currentUser.getUser().isEmpty() || !currentUser.getUser().get().getDAdmin()) {
            return Result.failure(ErrorCollection.builder().addError(TypedError
                    .forbidden(messages.getMessage("errors.access.denied", null, locale))).build());
        }
        return Result.success(null);
    }

    private ClusterStateDto getCachedClusterState() {
        boolean isLeader = clusterManager.isLeaderCached().orElse(false);
        boolean isFollower = clusterManager.isLeaderCached().map(v -> !v).orElse(false);
        boolean isMember = clusterManager.isMemberCached().orElse(false);
        ClusterNodeDto leader = clusterManager.getLeaderCached().map(this::toNodeState).orElse(null);
        Set<ClusterNodeDto> members = clusterManager.getMembersCached().map(nodes -> nodes.stream()
                .map(this::toNodeState).collect(Collectors.toSet())).orElse(Set.of());
        return new ClusterStateDto(isLeader, isFollower, isMember, leader, members);
    }

    private ClusterNodeDto toNodeState(NodeInfo nodeInfo) {
        return new ClusterNodeDto(nodeInfo.getUuid(), nodeInfo.getTransientFqdn(), nodeInfo.getPersistentFqdn(),
                nodeInfo.getHostIps(), nodeInfo.getVersion());
    }

}
