package ru.yandex.intranet.d.web.controllers.admin.folders

import io.swagger.v3.oas.annotations.Operation
import io.swagger.v3.oas.annotations.Parameter
import io.swagger.v3.oas.annotations.media.Content
import io.swagger.v3.oas.annotations.media.Schema
import io.swagger.v3.oas.annotations.responses.ApiResponse
import io.swagger.v3.oas.annotations.responses.ApiResponses
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.DeleteMapping
import org.springframework.web.bind.annotation.PathVariable
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.PutMapping
import org.springframework.web.bind.annotation.RequestBody
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RestController
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.services.folders.FolderService
import ru.yandex.intranet.d.util.response.Responses
import ru.yandex.intranet.d.web.errors.Errors
import ru.yandex.intranet.d.web.model.ErrorCollectionDto
import ru.yandex.intranet.d.web.model.folders.FrontFolderDto
import ru.yandex.intranet.d.web.model.folders.FrontFolderInputDto
import ru.yandex.intranet.d.web.security.Auth
import ru.yandex.intranet.d.web.security.roles.UserRole
import java.security.Principal
import java.util.*

@UserRole
@RestController
@RequestMapping("/admin/folders")
open class AdminFoldersController(
    private val folderService: FolderService
) {

    @Operation(summary = "Create folder.")
    @ApiResponses(
        ApiResponse(
            responseCode = "200", description = "Created folder data.", content = [Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE,
                schema = Schema(implementation = FrontFolderDto::class)
            )]
        ),
        ApiResponse(
            responseCode = "403", description = "Access denied.", content = [Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE,
                schema = Schema(implementation = ErrorCollectionDto::class)
            )]
        ),
        ApiResponse(
            responseCode = "422", description = "Error during validation.", content = [Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE,
                schema = Schema(implementation = ErrorCollectionDto::class)
            )]
        )
    )
    @PostMapping
    open fun createFolder(
        principal: Principal,
        locale: Locale,
        @RequestBody folderInput: FrontFolderInputDto
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return folderService.createFolder(folderInput, currentUser, locale)
            .map { it.match(Responses::okJson, Errors::toResponse) }
    }


    @Operation(summary = "Modify folder.")
    @ApiResponses(
        ApiResponse(
            responseCode = "200", description = "Updated folder data.", content = [Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE,
                schema = Schema(implementation = FrontFolderDto::class)
            )]
        ),
        ApiResponse(
            responseCode = "403", description = "Access denied.", content = [Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE,
                schema = Schema(implementation = ErrorCollectionDto::class)
            )]
        ),
        ApiResponse(
            responseCode = "404", description = "Folder not found.", content = [Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE,
                schema = Schema(implementation = ErrorCollectionDto::class)
            )]
        ),
        ApiResponse(
            responseCode = "409", description = "Version mismatch.", content = [Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE,
                schema = Schema(implementation = ErrorCollectionDto::class)
            )]
        ),
        ApiResponse(
            responseCode = "422", description = "Error during validation.", content = [Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE,
                schema = Schema(implementation = ErrorCollectionDto::class)
            )]
        )
    )
    @PutMapping(value = ["/{id}"])
    open fun updateFolder(
        @Parameter(description = "Folder id.", required = true)
        @PathVariable id: String,
        principal: Principal,
        locale: Locale,
        @RequestBody folderInput: FrontFolderInputDto
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return folderService.updateFolder(id, folderInput, currentUser, locale)
            .map { it.match(Responses::okJson, Errors::toResponse) }
    }


    @Operation(summary = "Delete folder.")
    @ApiResponses(
        ApiResponse(
            responseCode = "200", description = "Deleted folder data.", content = [Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE, schema = Schema(implementation = FrontFolderDto::class)
            )]
        ),
        ApiResponse(
            responseCode = "403", description = "Access denied.", content = [Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE,
                schema = Schema(implementation = ErrorCollectionDto::class)
            )]
        ),
        ApiResponse(
            responseCode = "423", description = "Deletion of this folder is not allowed.", content = [Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE,
                schema = Schema(implementation = ErrorCollectionDto::class)
            )]
        )
    )
    @DeleteMapping(value = ["/{id}"])
    open fun deleteFolder(
        @Parameter(description = "Folder id.", required = true)
        @PathVariable id: String,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return folderService.deleteFolder(id, currentUser, locale)
            .map { it.match(Responses::okJson, Errors::toResponse) }
    }
}
