package ru.yandex.intranet.d.web.controllers.admin.maintenance;

import java.security.Principal;
import java.time.Clock;
import java.util.Locale;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;
import ru.yandex.intranet.d.services.gc.QuotasProvisionsGCService;
import ru.yandex.intranet.d.services.transfer.RefreshTransferRequestService;
import ru.yandex.intranet.d.services.usage.YpUsageSyncService;
import ru.yandex.intranet.d.services.usage.YtUsageSyncService;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserRole;

/**
 * Maintenance controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserRole
@RestController
@RequestMapping("/admin/maintenance")
public class MaintenanceController {

    private final MessageSource messages;
    private final QuotasProvisionsGCService quotasProvisionsGCService;
    private final RefreshTransferRequestService refreshTransferRequestService;
    private final YtUsageSyncService ytUsageSyncService;
    private final YpUsageSyncService ypUsageSyncService;

    public MaintenanceController(@Qualifier("messageSource") MessageSource messages,
                                 QuotasProvisionsGCService quotasProvisionsGCService,
                                 RefreshTransferRequestService refreshTransferRequestService,
                                 YtUsageSyncService ytUsageSyncService,
                                 YpUsageSyncService ypUsageSyncService) {
        this.messages = messages;
        this.quotasProvisionsGCService = quotasProvisionsGCService;
        this.refreshTransferRequestService = refreshTransferRequestService;
        this.ytUsageSyncService = ytUsageSyncService;
        this.ypUsageSyncService = ypUsageSyncService;
    }

    @Operation(summary = "Run GC for quotas and provisions.")
    @ApiResponses(@ApiResponse(responseCode = "204", description = "GC was successfully executed."))
    @PostMapping(value = "/_gcQuotasProvisions", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> gcQuotasProvisions(Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        if (currentUser.getUser().isEmpty() || !currentUser.getUser().get().getDAdmin()) {
            return Mono.just(Errors.toResponse(ErrorCollection.builder().addError(TypedError
                    .forbidden(messages.getMessage("errors.access.denied", null, locale))).build()));
        }
        return quotasProvisionsGCService.gcQuotasProvisions(Clock.systemUTC())
                .thenReturn(ResponseEntity.noContent().build());
    }

    @Operation(summary = "Refresh transfer requests responsible.")
    @ApiResponses(@ApiResponse(responseCode = "204", description = "Transfer requests responsible refreshed."))
    @PostMapping(value = "/_refreshTransferRequests", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> refreshTransferRequests(Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        if (currentUser.getUser().isEmpty() || !currentUser.getUser().get().getDAdmin()) {
            return Mono.just(Errors.toResponse(ErrorCollection.builder().addError(TypedError
                    .forbidden(messages.getMessage("errors.access.denied", null, locale))).build()));
        }
        return refreshTransferRequestService.refresh().thenReturn(ResponseEntity.noContent().build());
    }

    @Operation(summary = "Sync YT usage.")
    @ApiResponses(@ApiResponse(responseCode = "204", description = "YT usage sync was successfully launched."))
    @PostMapping(value = "/_syncYtUsage", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> syncYtUsage(Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        return ytUsageSyncService.doSyncWithValidationMono(currentUser, locale, Clock.systemUTC())
                .map(r -> r.match(entity -> ResponseEntity.noContent().build(), Errors::toResponse));
    }

    @Operation(summary = "Sync YP usage.")
    @ApiResponses(@ApiResponse(responseCode = "204", description = "YP usage sync was successfully launched."))
    @PostMapping(value = "/_syncYpUsage", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> syncYpUsage(Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        return ypUsageSyncService.doSyncWithValidationMono(currentUser, locale, Clock.systemUTC())
                .map(r -> r.match(entity -> ResponseEntity.noContent().build(), Errors::toResponse));
    }

}
