package ru.yandex.intranet.d.web.controllers.admin.providers;

import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.BillingMeta;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.providers.RelatedResourceMapping;
import ru.yandex.intranet.d.services.admin.AdminService;
import ru.yandex.intranet.d.services.admin.JobStatus;
import ru.yandex.intranet.d.services.providers.ProvidersService;
import ru.yandex.intranet.d.util.AggregationAlgorithmHelper;
import ru.yandex.intranet.d.util.AggregationSettingsHelper;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.FeatureStateDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.folders.FolderDto;
import ru.yandex.intranet.d.web.model.folders.FrontFolderDto;
import ru.yandex.intranet.d.web.model.providers.ExternalAccountUrlTemplateDto;
import ru.yandex.intranet.d.web.model.providers.FullProviderDto;
import ru.yandex.intranet.d.web.model.providers.ProviderCreateDto;
import ru.yandex.intranet.d.web.model.providers.ProviderPutDto;
import ru.yandex.intranet.d.web.model.providers.ProviderUISettingsDto;
import ru.yandex.intranet.d.web.model.providers.RelatedResourceMappingDto;
import ru.yandex.intranet.d.web.model.quotas.ClearedQuotaDto;
import ru.yandex.intranet.d.web.model.quotas.QuotaSetDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserRole;

/**
 * Providers admin controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserRole
@RestController
@RequestMapping("/admin/providers")
public class AdminProvidersController {

    private final ProvidersService providersService;
    private final AdminService adminService;

    public AdminProvidersController(
            ProvidersService providersService,
            AdminService adminService
    ) {
        this.providersService = providersService;
        this.adminService = adminService;
    }

    @Operation(summary = "Get one provider by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested provider.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FullProviderDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ProviderModel>> result = providersService.getById(id, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toProvider(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Get one providers page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested providers page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FullProviderPageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPage(
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Page<ProviderModel>>> result = providersService
                .getPage(new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p)), Errors::toResponse));
    }

    @Operation(summary = "Create new provider.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Created provider.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FullProviderDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> create(
            @Parameter(description = "Provider to create", required = true)
            @RequestBody ProviderCreateDto provider,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ProviderModel>> result = providersService.create(provider, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toProvider(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Update existing provider.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated provider.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FullProviderDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PutMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> put(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Current provider version", required = true)
            @RequestParam("version") Long version,
            @Parameter(description = "Updated provider", required = true)
            @RequestBody ProviderPutDto provider,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ProviderModel>> result = providersService
                .put(id, version, provider, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toProvider(entity)), Errors::toResponse));
    }

    @Operation(summary = "Clean all quotas by resource type.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Quotas clean job started. Returned job ID",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ClearedQuotaDto.class))),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "Provider or Resource type or quotas not found",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "400", description = "Provider in unsuitable state",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))
    })
    @DeleteMapping(value = "/quotas/{providerId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> cleanQuotas(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Resource type id")
            @RequestParam(name = "resourceTypeId", required = false) String resourceTypeId,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ClearedQuotaDto>> result = adminService.runCleanQuotasMono(
                providerId,
                resourceTypeId,
                Tenants.getTenantId(currentUser),
                currentUser,
                locale
        );
        return result.mapNotNull(r -> r.match(Responses::okJson, Errors::toResponse));
    }

    @Operation(summary = "Get job status by job ID.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Job status",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = JobStatus.class))),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "Job not found",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(path = "/job/{jobId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getJobStatus(
            @Parameter(description = "Job ID", required = true)
            @PathVariable("jobId") String jobId,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        return Mono.justOrEmpty(
                adminService.getJobStatus(jobId, currentUser, locale)
                        .match(Responses::okJson, Errors::toResponse));
    }

    @Operation(summary = "Set quota from provided by provider.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Quota set successfully.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = QuotaSetDto.class))),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "Provider not found by id",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/{id}/_putProvisionsAsQuotas", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> putProvisionsAsQuota(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Void>> result = providersService.putProvisionsAsQuota(id, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.noContent(), Errors::toResponse));
    }

    @Operation(summary = "Create reserve folder in provider service.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Create successfully.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FolderDto.class))),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "Provider not found by id",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "Folder already exists.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/{id}/reserveFolder", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> createReserveFolder(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<FolderModel>> result = providersService.createReserveFolder(id, currentUser, locale);
        return result.map(r -> r.match(
                folder -> ResponseEntity.ok().contentType(MediaType.APPLICATION_JSON).body(FrontFolderDto.from(folder)),
                Errors::toResponse));
    }

    private PageDto<FullProviderDto> toPage(Page<ProviderModel> p) {
        return new PageDto<>(p.getItems().stream().map(this::toProvider)
                .collect(Collectors.toList()), p.getContinuationToken().orElse(null));
    }

    private FullProviderDto toProvider(ProviderModel model) {
        return new FullProviderDto(model.getId(), model.getVersion(),
                model.getNameEn(), model.getNameRu(),
                model.getDescriptionEn(), model.getDescriptionRu(),
                model.getRestApiUri().orElse(null),
                model.getGrpcApiUri().orElse(null),
                model.getSourceTvmId(),
                model.getDestinationTvmId(),
                model.getServiceId(), model.isReadOnly(),
                model.isMultipleAccountsPerFolder(),
                model.isAccountTransferWithQuota(),
                model.isManaged(),
                model.getKey(),
                model.getAccountsSettings().isDisplayNameSupported(),
                model.getAccountsSettings().isKeySupported(),
                model.getAccountsSettings().isDeleteSupported(),
                model.getAccountsSettings().isSoftDeleteSupported(),
                model.getAccountsSettings().isMoveSupported(),
                model.getAccountsSettings().isRenameSupported(),
                model.isImportAllowed(),
                model.isAccountsSpacesSupported(),
                model.getAccountsSettings().isPerAccountVersionSupported(),
                model.getAccountsSettings().isPerProvisionVersionSupported(),
                model.getAccountsSettings().isPerAccountLastUpdateSupported(),
                model.getAccountsSettings().isPerProvisionLastUpdateSupported(),
                model.getAccountsSettings().isOperationIdDeduplicationSupported(),
                model.getAccountsSettings().isSyncCoolDownDisabled(),
                model.getAccountsSettings().isRetryCoolDownDisabled(),
                model.getAccountsSettings().getAccountsSyncPageSize(),
                model.getAccountsSettings().isMoveProvisionSupported(),
                model.isSyncEnabled(),
                model.isGrpcTlsOn(),
                model.getBillingMeta().flatMap(BillingMeta::getMeteringKey).orElse(null),
                model.getRelatedResourcesByResourceId().map(this::toRelatedResourceMappingDto).orElse(null),
                model.getTrackerComponentId(),
                model.getReserveFolderId().orElse(null),
                model.hasDefaultQuotas(),
                FeatureStateDto.fromModel(model.isAllocatedSupported().orElse(null)),
                AggregationSettingsHelper.toAggregationSettingsDto(model.getAggregationSettings().orElse(null)),
                AggregationAlgorithmHelper.toAggregationAlgorithmDto(model.getAggregationAlgorithm().orElse(null)),
                model.getUiSettings().map(ProviderUISettingsDto::new).orElse(null),
                model.getAccountsSettings().getExternalAccountUrlTemplates().stream()
                        .map(ExternalAccountUrlTemplateDto::new).toList(),
                FeatureStateDto.fromModel(model.getAccountsSettings().getMultipleReservesAllowed().orElse(null))
        );
    }

    private Map<String, RelatedResourceMappingDto> toRelatedResourceMappingDto(
            Map<String, RelatedResourceMapping> relatedResourcesByResourceId) {
        return relatedResourcesByResourceId.entrySet().stream()
                .map(e -> Tuples.of(e.getKey(), RelatedResourceMappingDto.fromRelatedResourceMapping(e.getValue())))
                .collect(Collectors.toMap(Tuple2::getT1, Tuple2::getT2));
    }

    @Schema(description = "Providers page.")
    private static final class FullProviderPageDto extends PageDto<FullProviderDto> {
        private FullProviderPageDto(List<FullProviderDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }

}
