package ru.yandex.intranet.d.web.controllers.admin.retry;

import java.security.Principal;
import java.time.Clock;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.services.operations.OperationsRetryService;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.controllers.admin.retry.model.AccountCreateParamsDto;
import ru.yandex.intranet.d.web.controllers.admin.retry.model.AccountRenameParamsDto;
import ru.yandex.intranet.d.web.controllers.admin.retry.model.OperationChangesDto;
import ru.yandex.intranet.d.web.controllers.admin.retry.model.OperationChangesProvisionDto;
import ru.yandex.intranet.d.web.controllers.admin.retry.model.OperationDto;
import ru.yandex.intranet.d.web.controllers.admin.retry.model.OperationOrdersDto;
import ru.yandex.intranet.d.web.controllers.admin.retry.model.OperationSourceDto;
import ru.yandex.intranet.d.web.controllers.admin.retry.model.OperationTypeDto;
import ru.yandex.intranet.d.web.controllers.admin.retry.model.OperationsDto;
import ru.yandex.intranet.d.web.controllers.admin.retry.model.RequestStatusDto;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserRole;

/**
 * Retry admin controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserRole
@RestController
@RequestMapping("/admin/retry")
public class RetryController {

    private final MessageSource messages;
    private final OperationsRetryService retryService;

    public RetryController(@Qualifier("messageSource") MessageSource messages, OperationsRetryService retryService) {
        this.messages = messages;
        this.retryService = retryService;
    }

    @Operation(summary = "Run operations retry.")
    @ApiResponses(@ApiResponse(responseCode = "204", description = "Retry was successfully executed."))
    @PostMapping(value = "/_execute", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> doSync(Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        if (currentUser.getUser().isEmpty() || !currentUser.getUser().get().getDAdmin()) {
            return Mono.just(Errors.toResponse(ErrorCollection.builder().addError(TypedError
                    .forbidden(messages.getMessage("errors.access.denied", null, locale))).build()));
        }
        return retryService.retryOperations(Clock.systemUTC(), locale).thenReturn(ResponseEntity.noContent().build());
    }

    @Operation(summary = "Abort operation.")
    @ApiResponses({@ApiResponse(responseCode = "204", description = "Operation was successfully aborted."),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Operation not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/{id}/_abort", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> abortOperation(
            @Parameter(description = "Operation id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Comment", required = true)
            @RequestParam(value = "comment") String comment,
            Principal principal,
            Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        if (currentUser.getUser().isEmpty() || !currentUser.getUser().get().getDAdmin()
                || currentUser.getUser().get().getPassportUid().isEmpty()
                || currentUser.getUser().get().getPassportLogin().isEmpty()) {
            return Mono.just(Errors.toResponse(ErrorCollection.builder().addError(TypedError
                    .forbidden(messages.getMessage("errors.access.denied", null, locale))).build()));
        }
        Mono<Result<Void>> result = retryService.abortOperation(id, comment, currentUser, locale);
        return result.map(r -> r.match(entity -> ResponseEntity.noContent().build(), Errors::toResponse));
    }

    @Operation(summary = "Get all in progress operations.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "In progress operations list.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = OperationsDto.class)))})
    @GetMapping(value = "/_inProgress", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getInProgressOperations(Principal principal,
                                                           Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        if (currentUser.getUser().isEmpty() || !currentUser.getUser().get().getDAdmin()) {
            return Mono.just(Errors.toResponse(ErrorCollection.builder().addError(TypedError
                    .forbidden(messages.getMessage("errors.access.denied", null, locale))).build()));
        }
        Mono<Result<List<AccountsQuotasOperationsModel>>> result = retryService.getInProgressOperations();
        return result.map(r -> r.match(entity -> Responses.okJson(toDto(entity, locale)),
                Errors::toResponse));
    }

    private OperationsDto toDto(List<AccountsQuotasOperationsModel> ops, Locale locale) {
        return new OperationsDto(ops.stream().map(op -> new OperationDto(op.getOperationId(),
                op.getLastRequestId().orElse(null),
                op.getCreateDateTime(),
                OperationSourceDto.from(op.getOperationSource()),
                OperationTypeDto.from(op.getOperationType()),
                op.getAuthorUserId(),
                op.getAuthorUserUid().orElse(null),
                op.getProviderId(),
                op.getAccountsSpaceId().orElse(null),
                op.getUpdateDateTime().orElse(null),
                op.getRequestStatus().map(RequestStatusDto::from).orElse(null),
                op.getFullErrorMessage()
                        .map(e -> e.getErrorCollection(locale).toErrorCollection())
                        .map(Errors::flattenErrors)
                        .orElse(op.getErrorMessage().orElse(null)),
                new OperationChangesDto(op.getRequestedChanges().getAccountId().orElse(null),
                        op.getRequestedChanges().getUpdatedProvisions().map(l -> l.stream()
                                .map(p -> new OperationChangesProvisionDto(p.getResourceId(), p.getAmount()))
                                .collect(Collectors.toList())).orElse(List.of()),
                        op.getRequestedChanges().getFrozenProvisions().map(l -> l.stream()
                                .map(p -> new OperationChangesProvisionDto(p.getResourceId(), p.getAmount()))
                                .collect(Collectors.toList())).orElse(List.of()),
                        op.getRequestedChanges().getAccountCreateParams().map(p -> new AccountCreateParamsDto(
                                p.getKey().orElse(null),
                                p.getDisplayName().orElse(null),
                                p.getFolderId(),
                                p.getAccountId()
                        )).orElse(null),
                        op.getRequestedChanges().getAccountRenameParams().map(p -> new AccountRenameParamsDto(
                                p.getDisplayName()
                        )).orElse(null),
                        op.getRequestedChanges().getDestinationFolderId().orElse(null),
                        op.getRequestedChanges().getDestinationAccountId().orElse(null),
                        op.getRequestedChanges().getUpdatedDestinationProvisions().map(l -> l.stream()
                                .map(p -> new OperationChangesProvisionDto(p.getResourceId(), p.getAmount()))
                                .collect(Collectors.toList())).orElse(List.of()),
                        op.getRequestedChanges().getFrozenDestinationProvisions().map(l -> l.stream()
                                .map(p -> new OperationChangesProvisionDto(p.getResourceId(), p.getAmount()))
                                .collect(Collectors.toList())).orElse(List.of())),
                new OperationOrdersDto(op.getOrders().getSubmitOrder(),
                        op.getOrders().getCloseOrder().orElse(null),
                        op.getOrders().getRestoreOrder().orElse(null),
                        op.getOrders().getDestinationSubmitOrder().orElse(null),
                        op.getOrders().getDestinationCloseOrder().orElse(null),
                        op.getOrders().getDestinationRestoreOrder().orElse(null)))
        ).collect(Collectors.toList()));
    }

}
