package ru.yandex.intranet.d.web.controllers.admin.settings

import io.swagger.v3.oas.annotations.Operation
import io.swagger.v3.oas.annotations.Parameter
import io.swagger.v3.oas.annotations.media.Content
import io.swagger.v3.oas.annotations.media.Schema
import io.swagger.v3.oas.annotations.responses.ApiResponse
import io.swagger.v3.oas.annotations.responses.ApiResponses
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.PutMapping
import org.springframework.web.bind.annotation.RequestBody
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RestController
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.services.settings.RuntimeSettingsAdminService
import ru.yandex.intranet.d.util.response.Responses
import ru.yandex.intranet.d.web.errors.Errors
import ru.yandex.intranet.d.web.model.settings.KnownProvidersDto
import ru.yandex.intranet.d.web.model.settings.RateLimiterSettingsDto
import ru.yandex.intranet.d.web.model.settings.YpUsageSyncSettingsDto
import ru.yandex.intranet.d.web.model.settings.YtUsageSyncSettingsDto
import ru.yandex.intranet.d.web.security.Auth
import ru.yandex.intranet.d.web.security.roles.UserRole
import java.security.Principal
import java.util.*

/**
 * Runtime settings admin controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserRole
@RestController
@RequestMapping("/admin/settings")
open class RuntimeSettingsAdminController(private val runtimeSettingsAdminService: RuntimeSettingsAdminService) {

    @Operation(summary = "Get known providers settings.")
    @ApiResponses(ApiResponse(responseCode = "200", description = "Known provider settings.",
        content = [Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
        schema = Schema(implementation = KnownProvidersDto::class))]))
    @GetMapping(value = ["/knownProviders"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun getKnownProviders(
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return runtimeSettingsAdminService.getKnownProvidersMono(currentUser,
            locale).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(summary = "Get YT usage sync settings.")
    @ApiResponses(ApiResponse(responseCode = "200", description = "YT usage sync settings.",
        content = [Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = YtUsageSyncSettingsDto::class))]))
    @GetMapping(value = ["/ytUsageSync"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun getYtUsageSync(
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return runtimeSettingsAdminService.getYtUsageSyncSettingsMono(currentUser,
            locale).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(summary = "Get YP usage sync settings.")
    @ApiResponses(ApiResponse(responseCode = "200", description = "YP usage sync settings.",
        content = [Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = YpUsageSyncSettingsDto::class))]))
    @GetMapping(value = ["/ypUsageSync"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun getYpUsageSync(
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return runtimeSettingsAdminService.getYpUsageSyncSettingsMono(currentUser,
            locale).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(summary = "Get rate limiter settings.")
    @ApiResponses(ApiResponse(responseCode = "200", description = "Rate limiter settings.",
        content = [Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = RateLimiterSettingsDto::class))]))
    @GetMapping(value = ["/rateLimiter"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun getRateLimiterSettings(
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return runtimeSettingsAdminService.getRateLimiterSettingsMono(currentUser,
            locale).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(summary = "Put known providers settings.")
    @ApiResponses(ApiResponse(responseCode = "200", description = "Known provider settings.",
        content = [Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = KnownProvidersDto::class))]))
    @PutMapping(value = ["/knownProviders"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun putKnownProviders(
        @Parameter(description = "Known provider settings.", required = true)
        @RequestBody request: KnownProvidersDto,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return runtimeSettingsAdminService.putKnownProvidersMono(request, currentUser,
            locale).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(summary = "Put YT usage sync settings.")
    @ApiResponses(ApiResponse(responseCode = "200", description = "YT usage sync settings.",
        content = [Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = YtUsageSyncSettingsDto::class))]))
    @PutMapping(value = ["/ytUsageSync"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun putYtUsageSync(
        @Parameter(description = "YT usage sync settings.", required = true)
        @RequestBody request: YtUsageSyncSettingsDto,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return runtimeSettingsAdminService.putYtUsageSyncSettingsMono(request, currentUser,
            locale).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(summary = "Put YP usage sync settings.")
    @ApiResponses(ApiResponse(responseCode = "200", description = "YP usage sync settings.",
        content = [Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = YpUsageSyncSettingsDto::class))]))
    @PutMapping(value = ["/ypUsageSync"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun putYpUsageSync(
        @Parameter(description = "YP usage sync settings.", required = true)
        @RequestBody request: YpUsageSyncSettingsDto,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return runtimeSettingsAdminService.putYpUsageSyncSettingsMono(request, currentUser,
            locale).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(summary = "Put rate limiter settings.")
    @ApiResponses(ApiResponse(responseCode = "200", description = "Rate limiter settings.",
        content = [Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = RateLimiterSettingsDto::class))]))
    @PutMapping(value = ["/rateLimiter"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun putRateLimiterSettings(
        @Parameter(description = "Rate limiter settings.", required = true)
        @RequestBody request: RateLimiterSettingsDto,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return runtimeSettingsAdminService.putRateLimiterSettingsMono(request, currentUser,
            locale).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

}
