package ru.yandex.intranet.d.web.controllers.admin.sync;

import java.security.Principal;
import java.time.Clock;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import com.yandex.ydb.table.transaction.TransactionMode;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.providers.ProvidersDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.services.sync.AccountsSyncService;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserRole;

/**
 * Sync admin controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserRole
@RestController
@RequestMapping("/admin/sync")
public class SyncController {

    private static final Logger LOG = LoggerFactory.getLogger(SyncController.class);

    private final MessageSource messages;
    private final YdbTableClient tableClient;
    private final ProvidersDao providersDao;
    private final AccountsSyncService accountsSyncService;

    public SyncController(@Qualifier("messageSource") MessageSource messages, YdbTableClient tableClient,
                          ProvidersDao providersDao, AccountsSyncService accountsSyncService) {
        this.messages = messages;
        this.tableClient = tableClient;
        this.providersDao = providersDao;
        this.accountsSyncService = accountsSyncService;
    }

    @Operation(summary = "Run accounts sync.")
    @ApiResponses(@ApiResponse(responseCode = "204", description = "Sync was successfully executed."))
    @PostMapping(value = "/_execute", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> doSync(Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        if (currentUser.getUser().isEmpty() || !currentUser.getUser().get().getDAdmin()) {
            return Mono.just(Errors.toResponse(ErrorCollection.builder().addError(TypedError
                    .forbidden(messages.getMessage("errors.access.denied", null, locale))).build()));
        }
        return execute(locale).thenReturn(ResponseEntity.noContent().build());
    }

    private Mono<Void> execute(Locale locale) {
        return getProviders().flatMapIterable(v -> v).concatMap(p -> syncOneProvider(p, locale)
                .doOnError(e -> LOG.error("Failed to sync provider " + p.getKey(), e))
                .onErrorResume(v -> Mono.empty())
        ).collectList().then();
    }

    private Mono<Void> syncOneProvider(ProviderModel provider, Locale locale) {
        return accountsSyncService.syncOneProvider(provider, locale, Clock.systemUTC()).then();
    }

    private Mono<List<ProviderModel>> getProviders() {
        return tableClient.usingSessionMonoRetryable(session -> session
                .usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, txSession -> providersDao
                        .getAllByTenant(txSession, Tenants.DEFAULT_TENANT_ID, false)
                        .map(r -> r.get().stream().filter(this::eligibleForSync).collect(Collectors.toList()))));
    }

    private boolean eligibleForSync(ProviderModel provider) {
        return (provider.getRestApiUri().isPresent() || provider.getGrpcApiUri().isPresent())
                && provider.isSyncEnabled() && !provider.isDeleted();
    }

}
