package ru.yandex.intranet.d.web.controllers.admin.units;

import java.security.Principal;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.units.GrammaticalCase;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.units.UnitsComparator;
import ru.yandex.intranet.d.services.units.UnitsEnsemblesService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.units.FullUnitDto;
import ru.yandex.intranet.d.web.model.units.FullUnitsEnsembleDto;
import ru.yandex.intranet.d.web.model.units.GrammaticalCaseDto;
import ru.yandex.intranet.d.web.model.units.UnitCreateDto;
import ru.yandex.intranet.d.web.model.units.UnitPutDto;
import ru.yandex.intranet.d.web.model.units.UnitsEnsembleCreateDto;
import ru.yandex.intranet.d.web.model.units.UnitsEnsemblePatchDto;
import ru.yandex.intranet.d.web.model.units.UnitsEnsemblePutDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserRole;

/**
 * Units ensembles admin controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserRole
@RestController
@RequestMapping("/admin/unitsEnsembles")
public class AdminUnitsEnsemblesController {

    private final UnitsEnsemblesService unitsEnsemblesService;

    public AdminUnitsEnsemblesController(UnitsEnsemblesService unitsEnsemblesService) {
        this.unitsEnsemblesService = unitsEnsemblesService;
    }

    @Operation(summary = "Get one units ensemble by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested units ensemble.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FullUnitsEnsembleDto.class))),
            @ApiResponse(responseCode = "404", description = "'Units ensemble not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Units ensemble id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<UnitsEnsembleModel>> result = unitsEnsemblesService.getById(id, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toFullUnitsEnsemble(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Get one unit by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested unit.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FullUnitsEnsembleDto.class))),
            @ApiResponse(responseCode = "404", description = "'Unit not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{ensembleId}/units/{unitId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOneUnit(
            @Parameter(description = "Units ensemble id", required = true)
            @PathVariable("ensembleId") String ensembleId,
            @Parameter(description = "Unit id", required = true)
            @PathVariable("unitId") String unitId,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<UnitModel>> result = unitsEnsemblesService.getUnitById(ensembleId, unitId, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toFullUnit(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Get one units ensembles page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested units ensembles page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FullUnitsEnsemblePageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPage(
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Page<UnitsEnsembleModel>>> result = unitsEnsemblesService
                .getPage(new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p)), Errors::toResponse));
    }

    @Operation(summary = "Create new units ensemble.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Created units ensemble.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FullUnitsEnsembleDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> create(
            @Parameter(description = "Units ensemble to create", required = true)
            @RequestBody UnitsEnsembleCreateDto unitsEnsemble,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<UnitsEnsembleModel>> result = unitsEnsemblesService.create(unitsEnsemble, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toFullUnitsEnsemble(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Update existing units ensemble.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated units ensemble.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FullUnitsEnsembleDto.class))),
            @ApiResponse(responseCode = "404", description = "'Units ensemble not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PutMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> put(
            @Parameter(description = "Units ensemble id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Current units ensemble version", required = true)
            @RequestParam("version") Long version,
            @Parameter(description = "Updated units ensemble", required = true)
            @RequestBody UnitsEnsemblePutDto unitsEnsemble,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<UnitsEnsembleModel>> result = unitsEnsemblesService
                .put(id, version, unitsEnsemble, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toFullUnitsEnsemble(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Patch existing units ensemble.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Patched units ensemble.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FullUnitsEnsembleDto.class))),
            @ApiResponse(responseCode = "404", description = "'Units ensemble not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PatchMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> patch(
            @Parameter(description = "Units ensemble id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Current units ensemble version", required = true)
            @RequestParam("version") Long version,
            @Parameter(description = "Updated units ensemble", required = true)
            @RequestBody UnitsEnsemblePatchDto unitsEnsemble,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<UnitsEnsembleModel>> result = unitsEnsemblesService
                .patch(id, version, unitsEnsemble, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toFullUnitsEnsemble(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Delete units ensemble.")
    @ApiResponses({@ApiResponse(responseCode = "204", description = "Units ensemble was successfully deleted."),
            @ApiResponse(responseCode = "404", description = "'Units ensemble not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @DeleteMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> delete(
            @Parameter(description = "Units ensemble id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Current units ensemble version", required = true)
            @RequestParam("version") Long version,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Void>> result = unitsEnsemblesService
                .delete(id, version, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.noContent(), Errors::toResponse));
    }

    @Operation(summary = "Create new unit.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated units ensemble.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FullUnitsEnsembleDto.class))),
            @ApiResponse(responseCode = "404", description = "'Units ensemble not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/{ensembleId}/units", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> createUnit(
            @Parameter(description = "Units ensemble id", required = true)
            @PathVariable("ensembleId") String ensembleId,
            @Parameter(description = "Current units ensemble version", required = true)
            @RequestParam("version") Long version,
            @Parameter(description = "Unit to create", required = true)
            @RequestBody UnitCreateDto unit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<UnitsEnsembleModel>> result = unitsEnsemblesService
                .createUnit(ensembleId, version, unit, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toFullUnitsEnsemble(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Update unit.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated units ensemble.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FullUnitsEnsembleDto.class))),
            @ApiResponse(responseCode = "404", description = "'Unit not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PutMapping(value = "/{ensembleId}/units/{unitId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> putUnit(
            @Parameter(description = "Units ensemble id", required = true)
            @PathVariable("ensembleId") String ensembleId,
            @Parameter(description = "Unit id", required = true)
            @PathVariable("unitId") String unitId,
            @Parameter(description = "Current units ensemble version", required = true)
            @RequestParam("version") Long version,
            @Parameter(description = "Updated unit", required = true)
            @RequestBody UnitPutDto unit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<UnitsEnsembleModel>> result = unitsEnsemblesService
                .putUnit(ensembleId, version, unitId, unit, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toFullUnitsEnsemble(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Delete unit.")
    @ApiResponses({@ApiResponse(responseCode = "204", description = "Unit successfully deleted."),
            @ApiResponse(responseCode = "404", description = "'Unit not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @DeleteMapping(value = "/{ensembleId}/units/{unitId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> deleteUnit(
            @Parameter(description = "Units ensemble id", required = true)
            @PathVariable("ensembleId") String ensembleId,
            @Parameter(description = "Unit id", required = true)
            @PathVariable("unitId") String unitId,
            @Parameter(description = "Current units ensemble version", required = true)
            @RequestParam("version") Long version,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<UnitsEnsembleModel>> result = unitsEnsemblesService
                .deleteUnit(ensembleId, version, unitId, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.noContent(), Errors::toResponse));
    }

    private PageDto<FullUnitsEnsembleDto> toPage(Page<UnitsEnsembleModel> p) {
        return new PageDto<>(p.getItems().stream().map(this::toFullUnitsEnsemble)
                .collect(Collectors.toList()), p.getContinuationToken().orElse(null));
    }

    private FullUnitsEnsembleDto toFullUnitsEnsemble(UnitsEnsembleModel model) {
        return new FullUnitsEnsembleDto(model.getId(), model.getVersion(), model.getNameEn(), model.getNameRu(),
                model.getDescriptionEn(), model.getDescriptionRu(), model.isFractionsAllowed(),
                model.getUnits().stream().sorted(UnitsComparator.INSTANCE)
                        .map(this::toFullUnit).collect(Collectors.toList()), model.getKey());
    }

    private FullUnitDto toFullUnit(UnitModel model) {
        return new FullUnitDto(model.getId(), model.getKey(), toCases(model.getShortNameSingularRu()),
                toCases(model.getShortNamePluralRu()), model.getShortNameSingularEn(),
                model.getLongNamePluralEn(), toCases(model.getLongNameSingularRu()),
                toCases(model.getLongNamePluralRu()), model.getLongNameSingularEn(),
                model.getLongNamePluralEn(), model.getBase(), model.getPower());
    }

    private Map<GrammaticalCaseDto, String> toCases(Map<GrammaticalCase, String> model) {
        Map<GrammaticalCaseDto, String> result = new HashMap<>();
        model.forEach((k, v) -> toCase(k).ifPresent(c -> result.put(c, v)));
        return result;
    }

    private Optional<GrammaticalCaseDto> toCase(GrammaticalCase grammaticalCase) {
        switch (grammaticalCase) {
            case NOMINATIVE:
                return Optional.of(GrammaticalCaseDto.NOMINATIVE);
            case GENITIVE:
                return Optional.of(GrammaticalCaseDto.GENITIVE);
            case DATIVE:
                return Optional.of(GrammaticalCaseDto.DATIVE);
            case ACCUSATIVE:
                return Optional.of(GrammaticalCaseDto.ACCUSATIVE);
            case INSTRUMENTAL:
                return Optional.of(GrammaticalCaseDto.INSTRUMENTAL);
            case PREPOSITIONAL:
                return Optional.of(GrammaticalCaseDto.PREPOSITIONAL);
            case UNKNOWN:
                return Optional.empty();
            default:
                throw new IllegalArgumentException("Unexpected grammatical case: " + grammaticalCase);
        }
    }

    @Schema(description = "Units ensembles page.")
    private static final class FullUnitsEnsemblePageDto extends PageDto<FullUnitsEnsembleDto> {
        private FullUnitsEnsemblePageDto(List<FullUnitsEnsembleDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }

}
