package ru.yandex.intranet.d.web.controllers.api.v1.accounts;

import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.services.accounts.AccountLogicService;
import ru.yandex.intranet.d.services.accounts.AccountService;
import ru.yandex.intranet.d.services.accounts.AccountsReadService;
import ru.yandex.intranet.d.services.accounts.model.AccountOperationResult;
import ru.yandex.intranet.d.services.accounts.model.AccountOperationResultStatus;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.accounts.AccountDto;
import ru.yandex.intranet.d.web.model.accounts.AccountOperationDto;
import ru.yandex.intranet.d.web.model.accounts.AccountReserveTypeDto;
import ru.yandex.intranet.d.web.model.accounts.CreateAccountDto;
import ru.yandex.intranet.d.web.model.accounts.PutAccountDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * Providers public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
public class ApiV1AccountsController {

    private final AccountsReadService accountsReadService;
    private final AccountService accountService;
    private final AccountLogicService accountLogicService;

    public ApiV1AccountsController(AccountsReadService accountsReadService, AccountService accountService,
                                   AccountLogicService accountLogicService) {
        this.accountsReadService = accountsReadService;
        this.accountService = accountService;
        this.accountLogicService = accountLogicService;
    }

    @Operation(summary = "Get one account by folder id and account id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested account.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = AccountDto.class))),
            @ApiResponse(responseCode = "404", description = "'Account not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/api/v1/folders/{folderId}/accounts/{accountId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Folder id", required = true)
            @PathVariable("folderId") String folderId,
            @Parameter(description = "Account id", required = true)
            @PathVariable("accountId") String accountId,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<AccountModel>> result = accountsReadService.getById(folderId, accountId, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toAccount(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Get one accounts page by folder id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested accounts page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = AccountsPageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Folder not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/api/v1/folders/{folderId}/accounts", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPageByFolder(
            @Parameter(description = "Folder id", required = true)
            @PathVariable("folderId") String folderId,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Page<AccountModel>>> result = accountsReadService
                .getForFolder(folderId, new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p)), Errors::toResponse));
    }

    @Operation(summary = "Get one accounts page by folder id and provider id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested accounts page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = AccountsPageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Folder not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/api/v1/folders/{folderId}/providers/{providerId}/accounts",
            produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPageByFolderAndProvider(
            @Parameter(description = "Folder id", required = true)
            @PathVariable("folderId") String folderId,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Page<AccountModel>>> result = accountsReadService
                .getForProvider(folderId, providerId, new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p)), Errors::toResponse));
    }

    @Operation(summary = "Create account in folder.")
    @ApiResponses({@ApiResponse(responseCode = "201", description = "Account creation operation result.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = AccountOperationDto.class))),
            @ApiResponse(responseCode = "202", description = "Account creation operation acceptance result.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = AccountOperationDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Invalid parameters' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/api/v1/folders/{folderId}/accounts", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> create(
            @Parameter(description = "Folder id", required = true)
            @PathVariable("folderId") String folderId,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey,
            @RequestBody CreateAccountDto request,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<AccountOperationResult>> result = accountLogicService
                .createAccountMono(folderId, request, currentUser, locale, idempotencyKey);
        return result.map(r -> r.match(entity -> {
            if (entity.getOperationStatus() == AccountOperationResultStatus.SUCCESS) {
                return Responses.createdJson(toAccountOperation(entity));
            } else {
                return Responses.acceptedJson(toAccountOperation(entity));
            }
        }, Errors::toResponse));
    }

    @Operation(summary = "Update account in folder.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Account update operation result.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = AccountOperationDto.class))),
            @ApiResponse(responseCode = "202", description = "Account update operation acceptance result.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = AccountOperationDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Invalid parameters' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PutMapping(value = "/api/v1/folders/{folderId}/accounts/{accountId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> update(
            @Parameter(description = "Folder id", required = true)
            @PathVariable("folderId") String folderId,
            @Parameter(description = "Account id", required = true)
            @PathVariable("accountId") String accountId,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey,
            @RequestBody PutAccountDto request,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<AccountOperationResult>> result = accountLogicService
                .putAccountMono(folderId, accountId, request, currentUser, locale, idempotencyKey);
        return result.map(r -> r.match(entity -> {
            if (entity.getOperationStatus() == AccountOperationResultStatus.SUCCESS) {
                return Responses.okJson(toAccountOperation(entity));
            } else {
                return Responses.acceptedJson(toAccountOperation(entity));
            }
        }, Errors::toResponse));
    }

    private AccountDto toAccount(AccountModel account) {
        AccountDto.Builder builder = AccountDto.builder();
        builder.id(account.getId());
        builder.externalId(account.getOuterAccountIdInProvider());
        account.getOuterAccountKeyInProvider().ifPresent(builder::externalKey);
        builder.version(account.getVersion());
        builder.folderId(account.getFolderId());
        builder.providerId(account.getProviderId());
        account.getDisplayName().ifPresent(builder::displayName);
        account.getAccountsSpacesId().ifPresent(builder::accountsSpaceId);
        builder.freeTier(account.isFreeTier());
        builder.reserveType(AccountReserveTypeDto.fromModel(account.getReserveType().orElse(null)));
        return builder.build();
    }

    private AccountOperationDto toAccountOperation(AccountOperationResult operation) {
        AccountOperationDto.Builder builder = AccountOperationDto.builder();
        builder.operationId(operation.getOperationId());
        builder.operationStatus(AccountOperationResultStatus.toDto(operation.getOperationStatus()));
        operation.getAccount().ifPresent(a -> builder.result(toAccount(a)));
        return builder.build();
    }

    private PageDto<AccountDto> toPage(Page<AccountModel> p) {
        return new PageDto<>(p.getItems().stream().map(this::toAccount)
                .collect(Collectors.toList()), p.getContinuationToken().orElse(null));
    }

    @Schema(description = "Accounts page.")
    public static final class AccountsPageDto extends PageDto<AccountDto> {
        private AccountsPageDto(List<AccountDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }

}
