package ru.yandex.intranet.d.web.controllers.api.v1.aggregation

import io.swagger.v3.oas.annotations.Operation
import io.swagger.v3.oas.annotations.Parameter
import io.swagger.v3.oas.annotations.media.Content
import io.swagger.v3.oas.annotations.media.Schema
import io.swagger.v3.oas.annotations.responses.ApiResponse
import io.swagger.v3.oas.annotations.responses.ApiResponses
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RequestBody
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RestController
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.services.aggregates.QueryAggregatesService
import ru.yandex.intranet.d.util.response.Responses
import ru.yandex.intranet.d.web.errors.Errors
import ru.yandex.intranet.d.web.model.ErrorCollectionDto
import ru.yandex.intranet.d.web.model.aggregation.api.FindSubtreeTotalApiRequestDto
import ru.yandex.intranet.d.web.model.aggregation.api.FindSubtreeTotalApiResponseDto
import ru.yandex.intranet.d.web.model.aggregation.api.RankSubtreeAmountsApiRequestDto
import ru.yandex.intranet.d.web.model.aggregation.api.RankSubtreeAmountsApiResponseDto
import ru.yandex.intranet.d.web.security.Auth
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole
import java.security.Principal
import java.util.*

/**
 * Aggregation public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/aggregation")
open class AggregationApiController(private val queryAggregatesService: QueryAggregatesService) {

    @Operation(summary = "Find subtree total.")
    @ApiResponses(ApiResponse(responseCode = "200", description = "Subtree total.", content = [Content(
        mediaType = MediaType.APPLICATION_JSON_VALUE,
        schema = Schema(implementation = FindSubtreeTotalApiResponseDto::class))]),
        ApiResponse(responseCode = "422", description = "'Validation failed' error response.", content = [Content(
            mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = ErrorCollectionDto::class))]))
    @PostMapping(value = ["/_findSubtreeTotal"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun findSubtreeTotal(
        @Parameter(description = "Find subtree total request.", required = true)
        @RequestBody request: FindSubtreeTotalApiRequestDto,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return queryAggregatesService.findSubtreeTotalApiMono(request, currentUser,
            locale).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(summary = "Rank subtree amounts.")
    @ApiResponses(ApiResponse(responseCode = "200", description = "Ranked subtree amounts.", content = [Content(
        mediaType = MediaType.APPLICATION_JSON_VALUE,
        schema = Schema(implementation = RankSubtreeAmountsApiResponseDto::class))]),
        ApiResponse(responseCode = "422", description = "'Validation failed' error response.", content = [Content(
            mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = ErrorCollectionDto::class))]))
    @PostMapping(value = ["/_rankSubtreeAmounts"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun rankSubtreeAmounts(
        @Parameter(description = "Rank subtree amounts request.", required = true)
        @RequestBody request: RankSubtreeAmountsApiRequestDto,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return queryAggregatesService.rankSubtreeAmountsApiMono(request, currentUser,
            locale).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

}
