package ru.yandex.intranet.d.web.controllers.api.v1.delivery;

import java.security.Principal;
import java.util.List;
import java.util.Locale;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.services.delivery.DeliveryDestinationService;
import ru.yandex.intranet.d.services.delivery.DeliveryService;
import ru.yandex.intranet.d.services.delivery.provide.DeliveryAndProvideService;
import ru.yandex.intranet.d.services.delivery.provide.DeliveryAndProvideStatusService;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.delivery.DeliveryDestinationRequestDto;
import ru.yandex.intranet.d.web.model.delivery.DeliveryDestinationResponseDto;
import ru.yandex.intranet.d.web.model.delivery.DeliveryRequestDto;
import ru.yandex.intranet.d.web.model.delivery.DeliveryResponseDto;
import ru.yandex.intranet.d.web.model.delivery.provide.DeliveryAndProvideRequestDto;
import ru.yandex.intranet.d.web.model.delivery.status.DeliveryStatusDto;
import ru.yandex.intranet.d.web.model.delivery.status.DeliveryStatusResponseDto;
import ru.yandex.intranet.d.web.security.roles.HardwareOrderServiceRole;

/**
 * Providers public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@RestController
@HardwareOrderServiceRole
@RequestMapping("/api/v1/delivery")
public class DeliveryController {

    private final DeliveryService deliveryService;
    private final DeliveryDestinationService deliveryDestinationService;
    private final DeliveryAndProvideService deliverAndProvideService;
    private final DeliveryAndProvideStatusService deliverAndProvideStatusService;

    public DeliveryController(DeliveryService deliveryService,
                              DeliveryAndProvideService deliverAndProvideService,
                              DeliveryDestinationService deliveryDestinationService,
                              DeliveryAndProvideStatusService deliverAndProvideStatusService) {
        this.deliveryService = deliveryService;
        this.deliverAndProvideService = deliverAndProvideService;
        this.deliveryDestinationService = deliveryDestinationService;
        this.deliverAndProvideStatusService = deliverAndProvideStatusService;
    }

    @Operation(summary = "Deliver new quotas.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Delivery result.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = DeliveryResponseDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/_deliver", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> create(
            @Parameter(description = "Delivery request.", required = true)
            @RequestBody DeliveryRequestDto request,
            Principal principal, Locale locale) {
        Mono<Result<DeliveryResponseDto>> result = deliveryService.deliver(request, locale);
        return result.map(r -> r.match(Responses::okJson, Errors::toResponse));
    }

    @Operation(summary = "Find delivery destinations.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Delivery destination.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = DeliveryDestinationResponseDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/_findDestinations", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> findDestinations(
            @Parameter(description = "Delivery destination request.", required = true)
            @RequestBody DeliveryDestinationRequestDto request,
            Principal principal, Locale locale) {
        Mono<Result<DeliveryDestinationResponseDto>> result = deliveryDestinationService
                .getDestinationsMono(request, locale);
        return result.map(r -> r.match(Responses::okJson, Errors::toResponse));
    }

    @Operation(summary = "Deliver new quotas and provide it.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Delivery status.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = DeliveryStatusDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/_provide", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> provide(
            @Parameter(description = "Delivery and provide request.", required = true)
            @RequestBody DeliveryAndProvideRequestDto request,
            Principal principal, Locale locale) {
        Mono<Result<DeliveryStatusDto>> result = deliverAndProvideService.deliverAndProvide(request, locale);
        return result.map(r -> r.match(Responses::okJson, Errors::toResponse));
    }

    @Operation(summary = "Get the statuses of delivery operations to the account.")
    @ApiResponses({
            @ApiResponse(responseCode = "200", description = "Requested statuses.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = DeliveryStatusResponseDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))
    })
    @PostMapping("/_status")
    public Mono<ResponseEntity<?>> getDeliveryOperationStatuses(
            @Parameter(description = "Delivery ids. Maximum 1000 per request.", required = true)
            @RequestBody List<String> deliveryIds,
            Principal principal,
            Locale locale
    ) {
        Mono<Result<DeliveryStatusResponseDto>> result = deliverAndProvideStatusService
                .getDeliveryOperationStatuses(deliveryIds, locale);
        return result.map(r -> r.match(Responses::okJson, Errors::toResponse));
    }
}
