package ru.yandex.intranet.d.web.controllers.api.v1.folders;

import java.math.BigDecimal;
import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.folders.FolderService;
import ru.yandex.intranet.d.services.quotas.ExpandedQuotas;
import ru.yandex.intranet.d.services.quotas.QuotasService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.folders.FolderDto;
import ru.yandex.intranet.d.web.model.quotas.FolderQuotaDto;
import ru.yandex.intranet.d.web.model.quotas.FolderResourceQuotaDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * Folders public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/folders")
public class ApiV1FoldersController {

    private final FolderService folderService;
    private final QuotasService quotasService;

    public ApiV1FoldersController(FolderService folderService, QuotasService quotasService) {
        this.folderService = folderService;
        this.quotasService = quotasService;
    }

    @Operation(summary = "Get one folder by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested folder.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FolderDto.class))),
            @ApiResponse(responseCode = "404", description = "'Folder not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Folder id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<FolderModel>> result = folderService.getFolder(id, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toFolder(entity)), Errors::toResponse));
    }

    @Operation(summary = "Get one folders page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested folders page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FolderPageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPage(
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            @Parameter(description = "'With deleted' flag.")
            @RequestParam(value = "withDeleted", required = false, defaultValue = "false") boolean withDeleted,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Page<FolderModel>>> result = folderService
                .listFolders(withDeleted, new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p)), Errors::toResponse));
    }

    @Operation(summary = "Get folders by ids.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested folders.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FolderPageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/_by_ids", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getFoldersById(
            @Parameter(description = "Folder ids.", required = true)
            @RequestParam("ids") List<String> ids,
            @Parameter(description = "'With deleted' flag.")
            @RequestParam(value = "withDeleted", required = false, defaultValue = "false") boolean withDeleted,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<List<FolderModel>>> result = folderService.getFolders(ids, withDeleted, currentUser, locale);
        return result.map(r -> r.match(fs -> Responses.okJson(toPage(Page.lastPage(fs))), Errors::toResponse));
    }


    @Operation(summary = "Get one folder quotas page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested folder quotas page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FolderQuotaPageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Folder not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}/quotas", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getQuotasPage(
            @Parameter(description = "Folder id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedQuotas<Page<QuotaModel>>>> result = quotasService
                .getPageForFolder(id, new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toQuotaPage(p)), Errors::toResponse));
    }

    @Operation(summary = "Get folder resource quota.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested folder resource quota.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FolderResourceQuotaDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Folder not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}/providers/{providerId}/resources/{resourceId}/quota",
            produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getResourceQuota(
            @Parameter(description = "Folder id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Resource id", required = true)
            @PathVariable("resourceId") String resourceId,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedQuotas<QuotaModel>>> result = quotasService
                .getQuotaForFolderAndResource(id, resourceId, providerId, currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toFolderResourceQuota(p)), Errors::toResponse));
    }

    private FolderDto toFolder(FolderModel folder) {
        return new FolderDto(folder.getId(), folder.getVersion(), folder.getServiceId(), folder.getFolderType(),
                folder.getDisplayName(), folder.getDescription().orElse(null), folder.isDeleted(), folder.getTags());
    }

    private PageDto<FolderDto> toPage(Page<FolderModel> p) {
        return new PageDto<>(p.getItems().stream().map(this::toFolder)
                .collect(Collectors.toList()), p.getContinuationToken().orElse(null));
    }

    private FolderResourceQuotaDto toFolderResourceQuota(ExpandedQuotas<QuotaModel> quota) {
        ResourceModel resource = quota.getResources().get(quota.getQuotas().getResourceId());
        UnitsEnsembleModel unitsEnsemble = quota.getUnitsEnsembles().get(resource.getUnitsEnsembleId());
        Tuple2<BigDecimal, UnitModel> convertedQuota = Units.convertToApi(quota.getQuotas().getQuota(), resource,
                unitsEnsemble);
        Tuple2<BigDecimal, UnitModel> convertedBalance = Units.convertToApi(quota.getQuotas().getBalance(), resource,
                unitsEnsemble);
        return new FolderResourceQuotaDto(convertedQuota.getT1(), convertedQuota.getT2().getKey(),
                convertedBalance.getT1(), convertedBalance.getT2().getKey());
    }

    private FolderQuotaDto toFolderQuota(QuotaModel quota, Map<String, ResourceModel> resources,
                                         Map<String, UnitsEnsembleModel> unitsEnsembles) {
        ResourceModel resource = resources.get(quota.getResourceId());
        UnitsEnsembleModel unitsEnsemble = unitsEnsembles.get(resource.getUnitsEnsembleId());
        Tuple2<BigDecimal, UnitModel> convertedQuota = Units.convertToApi(quota.getQuota(), resource, unitsEnsemble);
        Tuple2<BigDecimal, UnitModel> convertedBalance = Units.convertToApi(quota.getBalance(), resource,
                unitsEnsemble);
        return new FolderQuotaDto(quota.getResourceId(), convertedQuota.getT1(), convertedQuota.getT2().getKey(),
                convertedBalance.getT1(), convertedBalance.getT2().getKey(), quota.getProviderId());
    }

    private PageDto<FolderQuotaDto> toQuotaPage(ExpandedQuotas<Page<QuotaModel>> p) {
        return new PageDto<>(p.getQuotas().getItems().stream()
                .map(q -> toFolderQuota(q, p.getResources(), p.getUnitsEnsembles()))
                .collect(Collectors.toList()), p.getQuotas().getContinuationToken().orElse(null));
    }

    @Schema(description = "Folders page.")
    private static final class FolderPageDto extends PageDto<FolderDto> {
        private FolderPageDto(List<FolderDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }

    @Schema(description = "Folder quotas page.")
    private static final class FolderQuotaPageDto extends PageDto<FolderQuotaDto> {
        private FolderQuotaPageDto(List<FolderQuotaDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }

}
