package ru.yandex.intranet.d.web.controllers.api.v1.imports;

import java.security.Principal;
import java.util.Locale;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.services.imports.QuotasImportsService;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.imports.ImportDto;
import ru.yandex.intranet.d.web.model.imports.ImportResultDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * Quotas import public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/import")
public class ApiV1ImportsController {

    private final QuotasImportsService quotasImportsService;

    public ApiV1ImportsController(QuotasImportsService quotasImportsService) {
        this.quotasImportsService = quotasImportsService;
    }

    @Operation(summary = "Import quotas.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Quotas import result.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ImportResultDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/_importQuotas", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> importQuotas(
            @Parameter(description = "Quotas to import", required = true)
            @RequestBody ImportDto quotasToImport,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ImportResultDto>> result = quotasImportsService
                .importQuotas(quotasToImport, currentUser, locale);
        return result.map(r -> r.match(Responses::okJson, Errors::toResponse));
    }

}
