package ru.yandex.intranet.d.web.controllers.api.v1.operations;

import java.security.Principal;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsRequestLogModel;
import ru.yandex.intranet.d.services.operations.OperationsService;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.operations.OperationDto;
import ru.yandex.intranet.d.web.model.operations.OperationErrorsDto;
import ru.yandex.intranet.d.web.model.operations.OperationRequestLogDto;
import ru.yandex.intranet.d.web.model.operations.OperationStatusDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * Operations public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/operations")
public class OperationsController {

    private final OperationsService operationsService;

    public OperationsController(OperationsService operationsService) {
        this.operationsService = operationsService;
    }

    @Operation(summary = "Get one operation by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested operation.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = OperationDto.class))),
            @ApiResponse(responseCode = "404", description = "'Operation not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Operation id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<AccountsQuotasOperationsModel>> result = operationsService.getById(id, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toOperation(entity, locale)),
                Errors::toResponse));
    }

    private OperationDto toOperation(AccountsQuotasOperationsModel operation, Locale locale) {
        OperationDto.Builder builder = OperationDto.builder()
                .id(operation.getOperationId())
                .providerId(operation.getProviderId())
                .status(toStatus(operation))
                .createdAt(operation.getCreateDateTime());
        if (operation.getLogs() != null) {
            List<OperationRequestLogDto> logs = new ArrayList<>(operation.getLogs().size());
            for (AccountsQuotasOperationsRequestLogModel log : operation.getLogs()) {
                OperationRequestLogDto logDto = OperationRequestLogDto.builder()
                        .requestId(log.getRequestId())
                        .createdAt(log.getCreateDateTime())
                        .requestData(log.getRequestData())
                        .responseData(log.getResponseData())
                        .expireAt(log.getExpireAt())
                        .build();
                logs.add(logDto);
            }
            builder.requestLogs(logs);
        }
        operation.getAccountsSpaceId().ifPresent(builder::accountsSpaceId);
        toErrors(operation, builder, locale);
        return builder.build();
    }

    private OperationStatusDto toStatus(AccountsQuotasOperationsModel operation) {
        if (operation.getRequestStatus().isEmpty()) {
            return OperationStatusDto.IN_PROGRESS;
        }
        switch (operation.getRequestStatus().get()) {
            case WAITING:
                return OperationStatusDto.IN_PROGRESS;
            case OK:
                return OperationStatusDto.SUCCESS;
            case ERROR:
                return OperationStatusDto.FAILURE;
            default:
                throw new IllegalArgumentException("Unexpected operation status "
                        + operation.getRequestStatus().get());
        }
    }

    private void toErrors(AccountsQuotasOperationsModel operation, OperationDto.Builder builder, Locale locale) {
        if (operation.getFullErrorMessage().isPresent()) {
            String flattenError = Errors.flattenErrors(operation.getFullErrorMessage().get()
                    .getErrorCollection(locale).toErrorCollection());
            builder.failure(OperationErrorsDto.builder().addError(flattenError).build());
        } else {
            operation.getErrorMessage().ifPresent(e ->
                    builder.failure(OperationErrorsDto.builder().addError(e).build()));
        }
    }

}
