package ru.yandex.intranet.d.web.controllers.api.v1.providers;

import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.services.accounts.ReserveAccountsService;
import ru.yandex.intranet.d.services.providers.ProviderRelatedResourcesService;
import ru.yandex.intranet.d.services.providers.ProvidersService;
import ru.yandex.intranet.d.services.provisions.ReserveProvisionsService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.providers.ExternalAccountUrlTemplateDto;
import ru.yandex.intranet.d.web.model.providers.ProviderDto;
import ru.yandex.intranet.d.web.model.providers.ProviderExternalAccountUrlTemplateDto;
import ru.yandex.intranet.d.web.model.providers.ProviderRelatedResourcesSettingsResponseDto;
import ru.yandex.intranet.d.web.model.providers.ProviderReserveAccountsDto;
import ru.yandex.intranet.d.web.model.providers.ProviderSyncStatusDto;
import ru.yandex.intranet.d.web.model.providers.ProviderUISettingsDto;
import ru.yandex.intranet.d.web.model.providers.PutProviderRelatedResourcesSettingsRequestDto;
import ru.yandex.intranet.d.web.model.provisions.UpdateProviderReserveProvisionsRequestDto;
import ru.yandex.intranet.d.web.model.provisions.UpdateProviderReserveProvisionsResponseDto;
import ru.yandex.intranet.d.web.model.provisions.UpdateProviderReserveProvisionsStatusDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

import static ru.yandex.intranet.d.web.util.ModelDtoConverter.toProvider;
import static ru.yandex.intranet.d.web.util.ModelDtoConverter.toRelatedResources;

/**
 * Providers public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/providers")
public class ApiV1ProvidersController {

    private final ProvidersService providersService;
    private final ReserveProvisionsService reserveProvisionsService;
    private final ProviderRelatedResourcesService providerRelatedResourcesService;
    private final ReserveAccountsService reserveAccountsService;

    public ApiV1ProvidersController(ProvidersService providersService,
                                    ReserveProvisionsService reserveProvisionsService,
                                    ProviderRelatedResourcesService providerRelatedResourcesService,
                                    ReserveAccountsService reserveAccountsService) {
        this.providersService = providersService;
        this.reserveProvisionsService = reserveProvisionsService;
        this.providerRelatedResourcesService = providerRelatedResourcesService;
        this.reserveAccountsService = reserveAccountsService;
    }

    @Operation(summary = "Get one provider by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested provider.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ProviderDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ProviderModel>> result = providersService.getById(id, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toProvider(entity, locale)),
                Errors::toResponse));
    }

    @Operation(summary = "Get one providers page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested providers page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ProviderPageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPage(
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Page<ProviderModel>>> result = providersService
                .getPage(new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p, locale)), Errors::toResponse));
    }

    @Operation(summary = "Change provider read only mode.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated provider."),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/{id}/_readOnly", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> setReadOnly(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "New read only mode", required = true)
            @RequestParam(value = "readOnly") boolean readOnly,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ProviderModel>> result = providersService
                .setReadOnly(id, readOnly, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toProvider(entity, locale)), Errors::toResponse));
    }

    @Operation(summary = "Sync provider accounts.")
    @ApiResponses({@ApiResponse(responseCode = "204", description = "Provider synced."),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "Provider can't be synced",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/{id}/_doSync", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> doSync(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {

        YaUserDetails currentUser = Auth.details(principal);

        return providersService.doSync(id, currentUser, locale)
                .map(r -> r.match(entity -> Responses.noContent(), Errors::toResponse));
    }

    @Operation(summary = "Get accounts sync status and errors.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Sync status and errors.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ProviderSyncStatusDto.class))),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}/_syncStatus", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getSyncStatus(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String providerId,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        return providersService.getSyncStatus(providerId, currentUser, locale)
                  .map(r -> r.match(Responses::okJson, Errors::toResponse));
    }

    @Operation(summary = "Update provisions for reserve account.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Provision update result.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = UpdateProviderReserveProvisionsResponseDto.class))),
            @ApiResponse(responseCode = "202", description = "Provision update acceptance result.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = UpdateProviderReserveProvisionsResponseDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Invalid parameters' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Newer data found on provider side' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/{id}/_provideReserve",
            produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> provideReserve(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String id,
            @RequestBody UpdateProviderReserveProvisionsRequestDto request,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<UpdateProviderReserveProvisionsResponseDto>> result = reserveProvisionsService
                .provideReserveMono(id, request, idempotencyKey, currentUser, locale);
        return result.map(r -> r.match(entity -> {
            if (entity.getOperationStatus() == UpdateProviderReserveProvisionsStatusDto.SUCCESS) {
                return Responses.okJson(entity);
            } else {
                return Responses.acceptedJson(entity);
            }
        }, Errors::toResponse));
    }

    @Operation(summary = "Get provider related resources settings by provider id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested provider related resources settings.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ProviderRelatedResourcesSettingsResponseDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}/relatedResources", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getProviderRelatedResources(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ProviderModel>> result = providersService.getById(id, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toRelatedResources(entity)), Errors::toResponse));
    }

    @Operation(summary = "Update provider related resources settings by provider id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated provider related resources settings.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ProviderRelatedResourcesSettingsResponseDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PutMapping(value = "/{id}/relatedResources", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> putProviderRelatedResources(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Updated provider related resources settings", required = true)
            @RequestBody PutProviderRelatedResourcesSettingsRequestDto putDto,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ProviderModel>> result = providerRelatedResourcesService.put(id, putDto, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toRelatedResources(entity)), Errors::toResponse));
    }

    @Operation(summary = "Update provider UI settings by provider id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated provider UI settings.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ProviderUISettingsDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PutMapping(value = "/{id}/uiSettings", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> setProviderUISettings(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Updated provider UI settings", required = true)
            @RequestBody SetProviderUISettingsRequestDto putDto,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ProviderModel>> result = providersService.mergeUiSettings(
                id, putDto.version, putDto.uiSettings.toModel(), currentUser, locale
        );
        return result.map(r -> r.match(
                entity -> Responses.okJson(entity.getUiSettings().map(ProviderUISettingsDto::new).orElse(null)),
                Errors::toResponse
        ));
    }

    @Operation(summary = "Update provider external account url template by provider id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated provider external account url template.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ProviderExternalAccountUrlTemplateDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PutMapping(value = "/{id}/externalAccountUrlTemplate", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> setProviderExternalAccountUrlTemplate(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Updated provider external account url template", required = true)
            @RequestBody ProviderExternalAccountUrlTemplateDto putDto,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ProviderModel>> result = providersService.mergeExternalAccountUrlTemplate(
                id, putDto.getVersion(), putDto.getExternalAccountUrlTemplate(), currentUser, locale);
        return result.map(r -> r.match(
                entity -> Responses.okJson(
                        new ProviderExternalAccountUrlTemplateDto(
                                entity.getVersion(),
                                entity.getAccountsSettings().getExternalAccountUrlTemplates().stream()
                                        .map(ExternalAccountUrlTemplateDto::new).toList())),
                Errors::toResponse
        ));
    }

    @Operation(summary = "Get provider external account url template by provider id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Get provider external account url template.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ProviderExternalAccountUrlTemplateDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}/externalAccountUrlTemplate", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getProviderExternalAccountUrlTemplate(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ProviderModel>> result = providersService.getById(id, currentUser, locale);
        return result.map(r -> r.match(
                entity -> Responses.okJson(
                        new ProviderExternalAccountUrlTemplateDto(
                                entity.getVersion(),
                                entity.getAccountsSettings().getExternalAccountUrlTemplates().stream()
                                        .map(ExternalAccountUrlTemplateDto::new).toList())),
                Errors::toResponse
        ));
    }

    @Operation(summary = "Get provider reserve accounts.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Provider reserve accounts.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ProviderReserveAccountsDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}/_reserveAccounts", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getReserveAccounts(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ProviderReserveAccountsDto>> result = reserveAccountsService
                .findReserveAccountsApiMono(id, currentUser, locale);
        return result.map(r -> r.match(Responses::okJson, Errors::toResponse));
    }

    private PageDto<ProviderDto> toPage(Page<ProviderModel> p, Locale locale) {
        return new PageDto<>(p.getItems().stream().map(m -> toProvider(m, locale))
                .collect(Collectors.toList()), p.getContinuationToken().orElse(null));
    }

    @Schema(description = "Providers page.")
    private static final class ProviderPageDto extends PageDto<ProviderDto> {
        private ProviderPageDto(List<ProviderDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }

    @Schema(description = "Request for set provider UI settings.")
    public static final class SetProviderUISettingsRequestDto {
        private final Long version;
        private final ProviderUISettingsDto uiSettings;

        public SetProviderUISettingsRequestDto(Long version, ProviderUISettingsDto uiSettings) {
            this.version = version;
            this.uiSettings = uiSettings;
        }

        public Long getVersion() {
            return version;
        }

        public ProviderUISettingsDto getUiSettings() {
            return uiSettings;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            SetProviderUISettingsRequestDto that = (SetProviderUISettingsRequestDto) o;
            return Objects.equals(version, that.version) && Objects.equals(uiSettings, that.uiSettings);
        }

        @Override
        public int hashCode() {
            return Objects.hash(version, uiSettings);
        }

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }
    }
}
