package ru.yandex.intranet.d.web.controllers.api.v1.providers.resources;

import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.services.resources.AccountsSpacesService;
import ru.yandex.intranet.d.services.resources.ExpandedAccountsSpaces;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.resources.AccountsSpaceDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;
import ru.yandex.intranet.d.web.util.ModelDtoConverter;

/**
 * ApiV1AccountsSpacesController.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 18.01.2021
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/providers/{providerId}/accountsSpaces")
public class ApiV1AccountsSpacesController {
    private final AccountsSpacesService accountsSpacesService;

    public ApiV1AccountsSpacesController(
            AccountsSpacesService accountsSpacesService
    ) {
        this.accountsSpacesService = accountsSpacesService;
    }


    @Operation(summary = "Get one accounts space by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested accounts space.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = AccountsSpaceDto.class))),
            @ApiResponse(responseCode = "404", description = "'Accounts space not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Accounts space id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedAccountsSpaces<AccountSpaceModel>>> result = accountsSpacesService.getByIdExpanded(
                id, providerId, currentUser, locale
        );
        return result.map(r -> r.match(entity -> Responses.okJson(ModelDtoConverter.toDto(entity, locale)),
                Errors::toResponse));
    }

    @Operation(summary = "Get one accounts spaces page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested accounts spaces page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ApiV1AccountsSpacesController.AccountsSpacesPageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPage(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedAccountsSpaces<Page<AccountSpaceModel>>>> result = accountsSpacesService
                .getPageExpanded(providerId, new PageRequest(pageToken, limit), currentUser, locale, false);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p, locale)),
                Errors::toResponse));
    }

    private AccountsSpacesPageDto toPage(ExpandedAccountsSpaces<Page<AccountSpaceModel>> em, Locale locale) {
        return new AccountsSpacesPageDto(
                em.getAccountsSpaces().getItems().stream()
                        .map(m -> ModelDtoConverter.toDto(m, em.getSegmentations(), em.getSegments(), locale))
                        .collect(Collectors.toList()),
                em.getAccountsSpaces().getContinuationToken().orElse(null)
        );
    }

    @Schema(description = "Accounts spaces page.")
    public static class AccountsSpacesPageDto extends PageDto<AccountsSpaceDto> {
        public AccountsSpacesPageDto(List<AccountsSpaceDto> items, String nextPageToken) {
            super(items, nextPageToken);
        }
    }
}
