package ru.yandex.intranet.d.web.controllers.api.v1.providers.resources;

import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.resources.ExpandedResources;
import ru.yandex.intranet.d.services.resources.ResourcesService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.resources.InnerResourceSegmentDto;
import ru.yandex.intranet.d.web.model.resources.InnerResourceSegmentationDto;
import ru.yandex.intranet.d.web.model.resources.InnerResourceSegmentationSegmentDto;
import ru.yandex.intranet.d.web.model.resources.InnerResourceTypeDto;
import ru.yandex.intranet.d.web.model.resources.InnerResourceTypeSegmentsDto;
import ru.yandex.intranet.d.web.model.resources.ResourceDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;
import ru.yandex.intranet.d.web.util.ModelDtoConverter;

import static ru.yandex.intranet.d.web.util.ModelDtoConverter.toResourceSegmentation;

/**
 * Provider resources public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/providers/{providerId}/resources")
public class ApiV1ResourcesController {

    private final ResourcesService resourcesService;

    public ApiV1ResourcesController(
            ResourcesService resourcesService
    ) {
        this.resourcesService = resourcesService;
    }

    @Operation(summary = "Get one resource by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested resource.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceDto.class))),
            @ApiResponse(responseCode = "404", description = "'Resource not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Resource id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Include segmentations in response, true or false")
            @RequestParam(value = "withSegmentations", required = false) Boolean withSegmentations,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        boolean withSegmentationsValue = withSegmentations != null ? withSegmentations : false;
        Mono<Result<ExpandedResources<ResourceModel>>> result = resourcesService
                .getById(id, providerId, currentUser, locale, false, withSegmentationsValue);
        return result.map(r -> r.match(entity -> Responses.okJson(toResource(entity, withSegmentationsValue, locale)),
                Errors::toResponse));
    }

    @Operation(summary = "Get one resources page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested resources page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourcePageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPage(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            @Parameter(description = "Include segmentations in response, true or false")
            @RequestParam(value = "withSegmentations", required = false) Boolean withSegmentations,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        boolean withSegmentationsValue = withSegmentations != null ? withSegmentations : false;
        Mono<Result<ExpandedResources<Page<ResourceModel>>>> result = resourcesService
                .getPage(providerId, new PageRequest(pageToken, limit), currentUser, locale,
                        false, withSegmentationsValue);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p, withSegmentationsValue, locale)),
                Errors::toResponse));
    }

    @SuppressWarnings("checkstyle:ParameterNumber")
    @Operation(summary = "Get page of resources by provider id, resource type and segments ids")
    @ApiResponses({
            @ApiResponse(responseCode = "200", description = "Requested resources page.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ResourcePageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider/segment not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))
    })
    @GetMapping(value = "/filterByType/{resourceTypeId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPageByProviderResourceTypeAndSegments(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Resource type id", required = true)
            @PathVariable(value = "resourceTypeId") String resourceTypeId,
            @Parameter(description = "Resource segments ids")
            @RequestParam(value = "segmentId", required = false) Set<String> segmentsIds,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            @Parameter(description = "Include segmentations in response, true or false")
            @RequestParam(value = "withSegmentations", required = false) Boolean withSegmentations,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        boolean withSegmentationsValue = withSegmentations != null ? withSegmentations : false;
        Mono<Result<ExpandedResources<Page<ResourceModel>>>> result = resourcesService
                .getPageByProviderTypeAndSegments(providerId, resourceTypeId, segmentsIds,
                        new PageRequest(pageToken, limit), currentUser, locale, withSegmentationsValue, false);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p, withSegmentationsValue, locale)),
                Errors::toResponse));
    }

    private PageDto<ResourceDto> toPage(ExpandedResources<Page<ResourceModel>> p, boolean withSegmentations,
                                        Locale locale) {
        return new PageDto<>(p.getResources().getItems().stream()
                .map(i -> toResource(i, p.getResourceTypes(), p.getResourceSegmentations(),
                        p.getResourceSegments(), p.getUnitsEnsembles(), withSegmentations, locale))
                .collect(Collectors.toList()), p.getResources().getContinuationToken().orElse(null));
    }

    private ResourceDto toResource(ExpandedResources<ResourceModel> model, boolean withSegmentations,
                                   Locale locale) {
        return toResource(model.getResources(), model.getResourceTypes(), model.getResourceSegmentations(),
                model.getResourceSegments(), model.getUnitsEnsembles(), withSegmentations, locale);
    }

    private ResourceDto toResource(ResourceModel model,
                                   Map<String, ResourceTypeModel> resourceTypes,
                                   Map<String, ResourceSegmentationModel> resourceSegmentations,
                                   Map<String, ResourceSegmentModel> resourceSegments,
                                   Map<String, UnitsEnsembleModel> unitsEnsembles,
                                   boolean withSegmentations,
                                   Locale locale) {
        return new ResourceDto(model.getId(),
                model.getKey(),
                model.getProviderId(),
                model.getUnitsEnsembleId(),
                model.getVersion(),
                Locales.select(model.getNameEn(), model.getNameRu(), locale),
                Locales.select(model.getDescriptionEn(), model.getDescriptionRu(), locale),
                model.isReadOnly(),
                model.isOrderable(),
                model.isManaged(),
                model.isVirtual(),
                model.getResourceUnits().getAllowedUnitIds()
                        .stream().map(i -> unitsEnsembles.get(model.getUnitsEnsembleId()).unitById(i)
                        .map(UnitModel::getKey).orElse(null)).collect(Collectors.toSet()),
                unitsEnsembles.get(model.getUnitsEnsembleId())
                        .unitById(model.getResourceUnits().getDefaultUnitId())
                        .map(UnitModel::getKey).orElse(null),
                model.getResourceUnits().getProviderApiUnitId(),
                toSegmentations(model, resourceTypes, resourceSegmentations,
                        resourceSegments, withSegmentations, locale));
    }

    private InnerResourceTypeSegmentsDto toSegmentations(ResourceModel model,
                                                         Map<String, ResourceTypeModel> resourceTypes,
                                                         Map<String, ResourceSegmentationModel> resourceSegmentations,
                                                         Map<String, ResourceSegmentModel> resourceSegments,
                                                         boolean withSegmentations,
                                                         Locale locale) {
        if (!withSegmentations || model.getResourceTypeId() == null) {
            return null;
        }
        InnerResourceTypeDto resourceType = toResourceType(resourceTypes
                .get(model.getResourceTypeId()), locale);
        Set<InnerResourceSegmentationSegmentDto> segments = model.getSegments().stream().map(s -> {
                    InnerResourceSegmentationDto segmentation = toResourceSegmentation(resourceSegmentations
                            .get(s.getSegmentationId()), locale);
                    InnerResourceSegmentDto segment = ModelDtoConverter.toResourceSegment(resourceSegments
                            .get(s.getSegmentId()), locale);
                    return new InnerResourceSegmentationSegmentDto(segmentation, segment);
                }).collect(Collectors.toSet());
        return new InnerResourceTypeSegmentsDto(resourceType, segments);
    }

    private InnerResourceTypeDto toResourceType(ResourceTypeModel model, Locale locale) {
        return new InnerResourceTypeDto(model.getId(),
                model.getKey(),
                Locales.select(model.getNameEn(), model.getNameRu(), locale),
                Locales.select(model.getDescriptionEn(), model.getDescriptionRu(), locale));
    }

    @Schema(description = "Resources page.")
    private static final class ResourcePageDto extends PageDto<ResourceDto> {
        private ResourcePageDto(List<ResourceDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }

}
