package ru.yandex.intranet.d.web.controllers.api.v1.providers.resources.directory;

import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.services.resources.AccountsSpacesService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.providers.ProviderUISettingsDto;
import ru.yandex.intranet.d.web.model.resources.directory.InnerResourceSegmentationSegmentDto;
import ru.yandex.intranet.d.web.model.resources.directory.spaces.AccountsSpaceCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.spaces.AccountsSpaceDto;
import ru.yandex.intranet.d.web.model.resources.directory.spaces.AccountsSpacePutDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * ApiV1AccountsSpacesController.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 18.01.2021
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/providers/{providerId}/resourcesDirectory/accountsSpaces")
public class ApiV1AccountsSpacesDirectoryController {
    private final AccountsSpacesService accountsSpacesService;

    public ApiV1AccountsSpacesDirectoryController(
            AccountsSpacesService accountsSpacesService
    ) {
        this.accountsSpacesService = accountsSpacesService;
    }


    @Operation(summary = "Get one accounts space by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested accounts space.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = AccountsSpaceDto.class))),
            @ApiResponse(responseCode = "404", description = "'Accounts space not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Accounts space id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<AccountSpaceModel>> result = accountsSpacesService.getById(
                id, providerId, currentUser, locale
        );
        return result.map(r -> r.match(entity -> Responses.okJson(toDto(entity)), Errors::toResponse));
    }

    @Operation(summary = "Get one accounts spaces page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested accounts spaces page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation =
                            ApiV1AccountsSpacesDirectoryController.AccountsSpacesPageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPage(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Page<AccountSpaceModel>>> result = accountsSpacesService
                .getPage(providerId, new PageRequest(pageToken, limit), currentUser, locale, false);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p, locale)),
                Errors::toResponse));
    }

    @Operation(summary = "Create new accounts space.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Created accounts space.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = AccountsSpaceDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "409", description = "'Key already exists' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> create(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Resource to create", required = true)
            @RequestBody AccountsSpaceCreateDto accountsSpace,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<AccountSpaceModel>> result = accountsSpacesService.create(
                accountsSpace, providerId, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toDto(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Update existing accounts space.")
    @ApiResponses({
            @ApiResponse(responseCode = "200", description = "Updated accounts space.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = AccountsSpaceDto.class))),
            @ApiResponse(responseCode = "404", description = "'Accounts space not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PutMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> put(
            @Parameter(description = "Accounts space id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Current resource version", required = true)
            @RequestParam("version") Long version,
            @Parameter(description = "Updated Accounts space", required = true)
            @RequestBody AccountsSpacePutDto accountsSpace,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<AccountSpaceModel>> result = accountsSpacesService
                .put(id, providerId, version, accountsSpace, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toDto(entity)), Errors::toResponse));
    }

    @Operation(summary = "Delete existing accounts space.")
    @ApiResponses({
            @ApiResponse(responseCode = "204", description = "Accounts space was successfully deleted."),
            @ApiResponse(responseCode = "404", description = "'Accounts space not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @DeleteMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> delete(
            @Parameter(description = "Accounts space id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        // TODO Implement this
        return Mono.just(ResponseEntity.status(HttpStatus.NOT_IMPLEMENTED).build());
    }

    @Operation(summary = "Change accounts space read only mode.")
    @ApiResponses({
            @ApiResponse(responseCode = "200", description = "Updated accounts space.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = AccountsSpaceDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/{id}/_readOnly", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> setReadOnly(
            @Parameter(description = "Accounts space id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "New read only mode", required = true)
            @RequestParam(value = "readOnly") boolean readOnly,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<AccountSpaceModel>> result = accountsSpacesService
                .setReadOnly(id, providerId, readOnly, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toDto(entity)), Errors::toResponse));
    }

    private AccountsSpacesPageDto toPage(Page<AccountSpaceModel> p, Locale locale) {
        return new AccountsSpacesPageDto(
                p.getItems().stream()
                        .map(this::toDto)
                        .collect(Collectors.toList()),
                p.getContinuationToken().orElse(null)
        );
    }

    private AccountsSpaceDto toDto(
            AccountSpaceModel model
    ) {
        return new AccountsSpaceDto(
                model.getId(), // id
                model.getOuterKeyInProvider(), // key
                model.getProviderId(), // providerId
                model.getVersion(), // version
                model.getNameEn(), // nameEn
                model.getNameRu(), // nameRu
                model.getDescriptionEn(), // descriptionEn
                model.getDescriptionRu(), // descriptionRu
                model.isReadOnly(), // readOnly
                toSegmentations(model), // segment
                model.getUiSettings().map(ProviderUISettingsDto::new).orElse(null),
                model.isSyncEnabled()
        );
    }

    private Set<InnerResourceSegmentationSegmentDto> toSegmentations(AccountSpaceModel model) {
        return model.getSegments().stream().map(s -> new InnerResourceSegmentationSegmentDto(
                s.getSegmentationId(), s.getSegmentId())).collect(Collectors.toSet());
    }

    @Schema(description = "Accounts spaces page.")
    public static class AccountsSpacesPageDto extends PageDto<AccountsSpaceDto> {
        public AccountsSpacesPageDto(List<AccountsSpaceDto> items, String nextPageToken) {
            super(items, nextPageToken);
        }
    }
}
