package ru.yandex.intranet.d.web.controllers.api.v1.providers.resources.directory;

import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.services.resources.ExpandedResources;
import ru.yandex.intranet.d.services.resources.ResourcesService;
import ru.yandex.intranet.d.util.AggregationSettingsHelper;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.FeatureStateDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.resources.directory.InnerResourceSegmentationSegmentDto;
import ru.yandex.intranet.d.web.model.resources.directory.InnerResourceTypeSegmentsDto;
import ru.yandex.intranet.d.web.model.resources.directory.ResourceCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.ResourceDto;
import ru.yandex.intranet.d.web.model.resources.directory.ResourcePutDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * Provider resources directory public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/providers/{providerId}/resourcesDirectory/resources")
public class ApiV1ResourcesDirectoryController {

    private final ResourcesService resourcesService;

    public ApiV1ResourcesDirectoryController(ResourcesService resourcesService) {
        this.resourcesService = resourcesService;
    }

    @Operation(summary = "Get one resource by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested resource.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceDto.class))),
            @ApiResponse(responseCode = "404", description = "'Resource not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Resource id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedResources<ResourceModel>>> result = resourcesService
                .getById(id, providerId, currentUser, locale, true, false);
        return result.map(r -> r.match(entity -> Responses.okJson(toResource(entity.getResources())),
                Errors::toResponse));
    }

    @Operation(summary = "Get one resources page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested resources page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourcePageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPage(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedResources<Page<ResourceModel>>>> result = resourcesService
                .getPage(providerId, new PageRequest(pageToken, limit), currentUser, locale,
                        true, false);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p.getResources())),
                Errors::toResponse));
    }

    @SuppressWarnings("checkstyle:ParameterNumber")
    @Operation(summary = "Get one resources page.")
    @ApiResponses({
            @ApiResponse(responseCode = "200", description = "Requested resources page.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ResourcePageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider/segment not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/filterByType/{resourceTypeId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPageByProviderResourceTypeAndSegments(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Resource type id", required = true)
            @PathVariable(value = "resourceTypeId") String resourceTypeId,
            @Parameter(description = "Resource segments ids")
            @RequestParam(value = "segmentId", required = false) Set<String> segmentsIds,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            @Parameter(description = "Include segmentations in response, true or false")
            @RequestParam(value = "withSegmentations", required = false) Boolean withSegmentations,
            Principal principal, Locale locale
    ) {
        boolean withSegmentationsValue = withSegmentations != null ? withSegmentations : false;
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedResources<Page<ResourceModel>>>> result = resourcesService
                .getPageByProviderTypeAndSegments(providerId, resourceTypeId, segmentsIds,
                        new PageRequest(pageToken, limit), currentUser, locale, withSegmentationsValue, true);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p.getResources())), Errors::toResponse));
    }

    @Operation(summary = "Create new resource.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Created resource.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "409", description = "'Key already exists' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> create(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Resource to create", required = true)
            @RequestBody ResourceCreateDto resource,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ResourceModel>> result = resourcesService.create(resource, providerId, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toResource(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Update existing resource.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated resource.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceDto.class))),
            @ApiResponse(responseCode = "404", description = "'Resource not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PutMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> put(
            @Parameter(description = "Resource id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Current resource version", required = true)
            @RequestParam("version") Long version,
            @Parameter(description = "Updated resource", required = true)
            @RequestBody ResourcePutDto resource,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ResourceModel>> result = resourcesService
                .put(id, providerId, version, resource, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toResource(entity)), Errors::toResponse));
    }

    @Operation(summary = "Change resource read only mode.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated resource."),
            @ApiResponse(responseCode = "404", description = "'Resource not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/{id}/_readOnly", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> setReadOnly(
            @Parameter(description = "Resource id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "New read only mode", required = true)
            @RequestParam(value = "readOnly") boolean readOnly,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ResourceModel>> result = resourcesService
                .setReadOnly(id, providerId, readOnly, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toResource(entity)), Errors::toResponse));
    }

    @Operation(summary = "Delete existing resource.")
    @ApiResponses({@ApiResponse(responseCode = "204", description = "Resource was successfully deleted."),
            @ApiResponse(responseCode = "404", description = "'Resource not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @DeleteMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> delete(
            @Parameter(description = "Resource id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        // TODO Implement this
        return Mono.just(ResponseEntity.status(HttpStatus.NOT_IMPLEMENTED).build());
    }

    private PageDto<ResourceDto> toPage(Page<ResourceModel> p) {
        return new PageDto<>(p.getItems().stream().map(this::toResource)
                .collect(Collectors.toList()), p.getContinuationToken().orElse(null));
    }

    private ResourceDto toResource(ResourceModel model) {
        return new ResourceDto(model.getId(),
                model.getKey(),
                model.getProviderId(),
                model.getUnitsEnsembleId(),
                model.getVersion(),
                model.getNameEn(),
                model.getNameRu(),
                model.getDescriptionEn(),
                model.getDescriptionRu(),
                model.isReadOnly(),
                model.isOrderable(),
                model.isManaged(),
                model.getResourceUnits().getAllowedUnitIds(),
                model.getResourceUnits().getDefaultUnitId(),
                model.getResourceUnits().getProviderApiUnitId(),
                toSegmentations(model),
                model.getDefaultQuota().orElse(null),
                model.isVirtual(),
                FeatureStateDto.fromModel(model.isAllocatedSupported().orElse(null)),
                AggregationSettingsHelper.toAggregationSettingsDto(model.getAggregationSettings().orElse(null)));
    }

    private InnerResourceTypeSegmentsDto toSegmentations(ResourceModel model) {
        if (model.getSegments() == null) {
            return null;
        }
        Set<InnerResourceSegmentationSegmentDto> segments = model.getSegments().stream()
                .map(s -> new InnerResourceSegmentationSegmentDto(s.getSegmentationId(), s.getSegmentId()))
                .collect(Collectors.toSet());
        return new InnerResourceTypeSegmentsDto(model.getResourceTypeId(), segments);
    }

    @Schema(description = "Resources page.")
    private static final class ResourcePageDto extends PageDto<ResourceDto> {
        private ResourcePageDto(List<ResourceDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }

}
