package ru.yandex.intranet.d.web.controllers.api.v1.providers.resources.directory.segmentations;

import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.services.resources.segmentations.ResourceSegmentationsService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.resources.directory.segmentations.ResourceSegmentationCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.segmentations.ResourceSegmentationDto;
import ru.yandex.intranet.d.web.model.resources.directory.segmentations.ResourceSegmentationPutDto;
import ru.yandex.intranet.d.web.model.resources.directory.segmentations.SegmentationUISettingsDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * Resource segmentations public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/providers/{providerId}/resourcesDirectory/segmentations")
public class ApiV1ResourceSegmentationsController {

    private final ResourceSegmentationsService resourceSegmentationsService;

    public ApiV1ResourceSegmentationsController(ResourceSegmentationsService resourceSegmentationsService) {
        this.resourceSegmentationsService = resourceSegmentationsService;
    }

    @Operation(summary = "Get one resource segmentation by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested resource segmentation.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceSegmentationDto.class))),
            @ApiResponse(responseCode = "404", description = "'Resource segmentation not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Resource segmentation id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ResourceSegmentationModel>> result = resourceSegmentationsService
                .getById(id, providerId, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toResourceSegmentation(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Get one resource segmentations page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested resource segmentations page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceSegmentationPageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPage(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Page<ResourceSegmentationModel>>> result = resourceSegmentationsService
                .getPage(providerId, new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p)), Errors::toResponse));
    }

    @Operation(summary = "Create new resource segmentation.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Created resource segmentation.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceSegmentationDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "409", description = "'Key already exists' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> create(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Resource segmentation to create", required = true)
            @RequestBody ResourceSegmentationCreateDto resourceSegmentation,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ResourceSegmentationModel>> result = resourceSegmentationsService
                .create(resourceSegmentation, providerId, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toResourceSegmentation(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Update existing resource segmentation.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated resource segmentation.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceSegmentationDto.class))),
            @ApiResponse(responseCode = "404", description = "'Resource segmentation not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PutMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> put(
            @Parameter(description = "Resource segmentation id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Current resource segmentation version", required = true)
            @RequestParam("version") Long version,
            @Parameter(description = "Updated resource segmentation", required = true)
            @RequestBody ResourceSegmentationPutDto resourceSegmentation,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ResourceSegmentationModel>> result = resourceSegmentationsService
                .put(id, providerId, version, resourceSegmentation, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toResourceSegmentation(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Delete existing resource segmentation.")
    @ApiResponses({@ApiResponse(responseCode = "204",
            description = "Resource segmentation was successfully deleted."),
            @ApiResponse(responseCode = "404", description = "'Resource segmentation not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @DeleteMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> delete(
            @Parameter(description = "Resource segmentation id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        // TODO Implement this
        return Mono.just(ResponseEntity.status(HttpStatus.NOT_IMPLEMENTED).build());
    }

    private PageDto<ResourceSegmentationDto> toPage(Page<ResourceSegmentationModel> p) {
        return new PageDto<>(p.getItems().stream().map(this::toResourceSegmentation)
                .collect(Collectors.toList()), p.getContinuationToken().orElse(null));
    }

    private ResourceSegmentationDto toResourceSegmentation(ResourceSegmentationModel model) {
        return new ResourceSegmentationDto(model.getId(), model.getProviderId(), model.getVersion(),
                model.getKey(), model.getNameEn(), model.getNameRu(), model.getDescriptionEn(),
                model.getDescriptionRu(), model.getGroupingOrder(),
                model.getUiSettings().map(SegmentationUISettingsDto::new).orElse(null)
        );
    }

    @Schema(description = "Resource segmentations page.")
    private static final class ResourceSegmentationPageDto extends PageDto<ResourceSegmentationDto> {
        private ResourceSegmentationPageDto(List<ResourceSegmentationDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }

}
