package ru.yandex.intranet.d.web.controllers.api.v1.providers.resources.directory.segmentations.segments;

import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.services.resources.segments.ResourceSegmentsService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.resources.directory.segments.ResourceSegmentCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.segments.ResourceSegmentDto;
import ru.yandex.intranet.d.web.model.resources.directory.segments.ResourceSegmentPutDto;
import ru.yandex.intranet.d.web.model.resources.directory.types.ResourceTypeDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * Resource segments public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/providers/{providerId}/resourcesDirectory/segmentations/{segmentationId}/segments")
public class ApiV1ResourceSegmentsController {

    private final ResourceSegmentsService resourceSegmentsService;

    public ApiV1ResourceSegmentsController(ResourceSegmentsService resourceSegmentsService) {
        this.resourceSegmentsService = resourceSegmentsService;
    }

    @Operation(summary = "Get one resource segment by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested resource segment.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceTypeDto.class))),
            @ApiResponse(responseCode = "404", description = "'Resource segment not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Segmentation id", required = true)
            @PathVariable("segmentationId") String segmentationId,
            @Parameter(description = "Resource segment id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ResourceSegmentModel>> result = resourceSegmentsService
                .getById(id, providerId, segmentationId, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toResourceSegment(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Get one resource types page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested resource segments page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceSegmentPageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPage(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Segmentation id", required = true)
            @PathVariable("segmentationId") String segmentationId,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Page<ResourceSegmentModel>>> result = resourceSegmentsService
                .getPage(providerId, segmentationId, new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p)), Errors::toResponse));
    }

    @Operation(summary = "Create new resource segment.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Created resource segment.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceSegmentDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "409", description = "'Key already exists' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> create(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Segmentation id", required = true)
            @PathVariable("segmentationId") String segmentationId,
            @Parameter(description = "Resource segment to create", required = true)
            @RequestBody ResourceSegmentCreateDto resourceSegment,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ResourceSegmentModel>> result = resourceSegmentsService
                .create(resourceSegment, providerId, segmentationId, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toResourceSegment(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Update existing resource segment.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated resource segment.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceSegmentDto.class))),
            @ApiResponse(responseCode = "404", description = "'Resource segment not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PutMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> put(
            @Parameter(description = "Resource segment id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Segmentation id", required = true)
            @PathVariable("segmentationId") String segmentationId,
            @Parameter(description = "Current resource segment version", required = true)
            @RequestParam("version") Long version,
            @Parameter(description = "Updated resource segment", required = true)
            @RequestBody ResourceSegmentPutDto resourceSegment,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ResourceSegmentModel>> result = resourceSegmentsService
                .put(id, providerId, segmentationId, version, resourceSegment, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toResourceSegment(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Delete existing resource segment.")
    @ApiResponses({@ApiResponse(responseCode = "204",
            description = "Resource segment was successfully deleted."),
            @ApiResponse(responseCode = "404", description = "'Resource segment not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @DeleteMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> delete(
            @Parameter(description = "Resource segment id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Segmentation id", required = true)
            @PathVariable("segmentationId") String segmentationId,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        // TODO Implement this
        return Mono.just(ResponseEntity.status(HttpStatus.NOT_IMPLEMENTED).build());
    }

    private PageDto<ResourceSegmentDto> toPage(Page<ResourceSegmentModel> p) {
        return new PageDto<>(p.getItems().stream().map(this::toResourceSegment)
                .collect(Collectors.toList()), p.getContinuationToken().orElse(null));
    }

    private ResourceSegmentDto toResourceSegment(ResourceSegmentModel model) {
        return new ResourceSegmentDto(model.getId(), model.getSegmentationId(), model.getVersion(),
                model.getKey(), model.getNameEn(), model.getNameRu(), model.getDescriptionEn(),
                model.getDescriptionRu(), model.getUncommon().orElse(false));
    }

    @Schema(description = "Resource segments page.")
    private static final class ResourceSegmentPageDto extends PageDto<ResourceSegmentDto> {
        private ResourceSegmentPageDto(List<ResourceSegmentDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }

}
