package ru.yandex.intranet.d.web.controllers.api.v1.providers.resources.directory.types;

import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.services.resources.types.ResourceTypesService;
import ru.yandex.intranet.d.util.AggregationSettingsHelper;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.resources.directory.types.ResourceTypeCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.types.ResourceTypeDto;
import ru.yandex.intranet.d.web.model.resources.directory.types.ResourceTypePatchDto;
import ru.yandex.intranet.d.web.model.resources.directory.types.ResourceTypePutDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * Resource types public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/providers/{providerId}/resourcesDirectory/resourceTypes")
public class ApiV1ResourceTypesController {

    private final ResourceTypesService resourceTypesService;

    public ApiV1ResourceTypesController(ResourceTypesService resourceTypesService) {
        this.resourceTypesService = resourceTypesService;
    }

    @Operation(summary = "Get one resource type by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested resource type.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceTypeDto.class))),
            @ApiResponse(responseCode = "404", description = "'Resource type not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Resource type id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ResourceTypeModel>> result = resourceTypesService.getById(id, providerId, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toResourceType(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Get one resource types page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested resource types page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceTypePageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPage(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Page<ResourceTypeModel>>> result = resourceTypesService
                .getPage(providerId, new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p)), Errors::toResponse));
    }

    @Operation(summary = "Create new resource type.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Created resource type.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceTypeDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "409", description = "'Key already exists' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> create(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Resource type to create", required = true)
            @RequestBody ResourceTypeCreateDto resourceType,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ResourceTypeModel>> result = resourceTypesService
                .create(resourceType, providerId, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toResourceType(entity)),
                Errors::toResponse));
    }

    @Operation(summary = "Update existing resource type.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated resource type.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceTypeDto.class))),
            @ApiResponse(responseCode = "404", description = "'Resource type not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PutMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> put(
            @Parameter(description = "Resource type id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Current resource type version", required = true)
            @RequestParam("version") Long version,
            @Parameter(description = "Updated resource type", required = true)
            @RequestBody ResourceTypePutDto resourceType,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ResourceTypeModel>> result = resourceTypesService
                .put(id, providerId, version, resourceType, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toResourceType(entity)), Errors::toResponse));
    }

    @Operation(summary = "Patch existing resource type.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Patched resource type.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ResourceTypeDto.class))),
            @ApiResponse(responseCode = "404", description = "'Resource type not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PatchMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> patch(
            @Parameter(description = "Resource type id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Current resource type version", required = true)
            @RequestParam("version") Long version,
            @Parameter(description = "Patched resource type", required = true)
            @RequestBody ResourceTypePatchDto resourceType,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ResourceTypeModel>> result = resourceTypesService
                .patch(id, providerId, version, resourceType, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toResourceType(entity)), Errors::toResponse));
    }

    @Operation(summary = "Delete existing resource type.")
    @ApiResponses({@ApiResponse(responseCode = "204", description = "Resource type was successfully deleted."),
            @ApiResponse(responseCode = "404", description = "'Resource type not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @DeleteMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> delete(
            @Parameter(description = "Resource type id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        // TODO Implement this
        return Mono.just(ResponseEntity.status(HttpStatus.NOT_IMPLEMENTED).build());
    }

    private PageDto<ResourceTypeDto> toPage(Page<ResourceTypeModel> p) {
        return new PageDto<>(p.getItems().stream().map(this::toResourceType)
                .collect(Collectors.toList()), p.getContinuationToken().orElse(null));
    }

    private ResourceTypeDto toResourceType(ResourceTypeModel model) {
        return new ResourceTypeDto(model.getId(), model.getProviderId(), model.getVersion(),
                model.getKey(), model.getNameEn(), model.getNameRu(), model.getDescriptionEn(),
                model.getDescriptionRu(), model.getUnitsEnsembleId(), model.getSortingOrder(),
                AggregationSettingsHelper.toAggregationSettingsDto(model.getAggregationSettings().orElse(null)));
    }

    @Schema(description = "Resource types page.")
    private static final class ResourceTypePageDto extends PageDto<ResourceTypeDto> {
        private ResourceTypePageDto(List<ResourceTypeDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }

}
