package ru.yandex.intranet.d.web.controllers.api.v1.provisions;

import java.math.BigDecimal;
import java.security.Principal;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.provisions.ProvisionsReadService;
import ru.yandex.intranet.d.services.quotas.ExpandedQuotas;
import ru.yandex.intranet.d.services.quotas.ProvisionLogicService;
import ru.yandex.intranet.d.services.quotas.ProvisionOperationResult;
import ru.yandex.intranet.d.services.quotas.ProvisionOperationResultStatus;
import ru.yandex.intranet.d.services.quotas.ProvisionService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.provisions.AccountProvisionDto;
import ru.yandex.intranet.d.web.model.provisions.UpdateProvisionOperationDto;
import ru.yandex.intranet.d.web.model.provisions.UpdateProvisionOperationStatusDto;
import ru.yandex.intranet.d.web.model.provisions.UpdateProvisionsDto;
import ru.yandex.intranet.d.web.model.quotas.ProvisionLiteDto;
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionsRequestDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * Provisions public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
public class ApiV1ProvisionsController {

    private final ProvisionsReadService provisionsReadService;
    private final ProvisionService provisionService;
    private final ProvisionLogicService provisionLogicService;

    public ApiV1ProvisionsController(ProvisionsReadService provisionsReadService, ProvisionService provisionService,
                                     ProvisionLogicService provisionLogicService) {
        this.provisionsReadService = provisionsReadService;
        this.provisionService = provisionService;
        this.provisionLogicService = provisionLogicService;
    }

    @Operation(summary = "Get one provision by folder account and provider resource.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested provision.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = AccountProvisionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Account not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value =
            "/api/v1/folders/{folderId}/providers/{providerId}/accounts/{accountId}/resources/{resourceId}/provision",
            produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Folder id", required = true)
            @PathVariable("folderId") String folderId,
            @Parameter(description = "Account id", required = true)
            @PathVariable("accountId") String accountId,
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Resource id", required = true)
            @PathVariable("resourceId") String resourceId,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedQuotas<AccountsQuotasModel>>> result = provisionsReadService.getOneProvision(folderId,
                accountId, providerId, resourceId, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toProvision(entity.getQuotas(),
                        entity.getResources(), entity.getUnitsEnsembles())), Errors::toResponse));
    }

    @Operation(summary = "Get one provisions page by folder account.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested provisions page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = AccountProvisionsPageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Account not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/api/v1/folders/{folderId}/accounts/{accountId}/provisions",
            produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPageByFolder(
            @Parameter(description = "Folder id", required = true)
            @PathVariable("folderId") String folderId,
            @Parameter(description = "Account id", required = true)
            @PathVariable("accountId") String accountId,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedQuotas<Page<AccountsQuotasModel>>>> result = provisionsReadService
                .getAccountProvisions(folderId, accountId, new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p.getQuotas(), p.getResources(),
                p.getUnitsEnsembles())), Errors::toResponse));
    }

    @Operation(summary = "Update provisions for account.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Provision update result.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = UpdateProvisionOperationDto.class))),
            @ApiResponse(responseCode = "202", description = "Provision update acceptance result.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = UpdateProvisionOperationDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Invalid parameters' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Newer data found on provider side' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/api/v1/folders/{folderId}/accounts/{accountId}/_provide",
            produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> provide(
            @Parameter(description = "Folder id", required = true)
            @PathVariable("folderId") String folderId,
            @Parameter(description = "Account id", required = true)
            @PathVariable("accountId") String accountId,
            @RequestBody UpdateProvisionsDto request,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ProvisionOperationResult>> result = provisionLogicService
                .updateProvisionMono(toProvisionRequest(folderId, accountId, request), currentUser, locale, true,
                        idempotencyKey);
        return result.map(r -> r.match(entity -> {
            if (entity.getStatus() == ProvisionOperationResultStatus.SUCCESS) {
                return Responses.okJson(toProvisionResponse(entity));
            } else {
                return Responses.acceptedJson(toProvisionResponse(entity));
            }
        }, Errors::toResponse));
    }

    private AccountProvisionDto toProvision(AccountsQuotasModel provision, Map<String, ResourceModel> resources,
                                            Map<String, UnitsEnsembleModel> unitsEnsembles) {
        ResourceModel resource = resources.get(provision.getResourceId());
        UnitsEnsembleModel unitsEnsemble = unitsEnsembles.get(resource.getUnitsEnsembleId());
        long provided = provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L;
        long allocated = provision.getAllocatedQuota() != null ? provision.getAllocatedQuota() : 0L;
        Tuple2<BigDecimal, UnitModel> convertedProvided = Units.convertToApi(provided, resource, unitsEnsemble);
        Tuple2<BigDecimal, UnitModel> convertedAllocated = Units.convertToApi(allocated, resource, unitsEnsemble);
        AccountProvisionDto.Builder builder = AccountProvisionDto.builder();
        builder.folderId(provision.getFolderId());
        builder.accountId(provision.getAccountId());
        builder.providerId(provision.getProviderId());
        builder.resourceId(provision.getResourceId());
        builder.provided(convertedProvided.getT1());
        builder.providedUnitKey(convertedProvided.getT2().getKey());
        builder.allocated(convertedAllocated.getT1());
        builder.allocatedUnitKey(convertedAllocated.getT2().getKey());
        return builder.build();
    }

    private PageDto<AccountProvisionDto> toPage(Page<AccountsQuotasModel> page,
                                                Map<String, ResourceModel> resources,
                                                Map<String, UnitsEnsembleModel> unitsEnsembles) {
        return new PageDto<>(page.getItems().stream().map(p -> toProvision(p, resources, unitsEnsembles))
                .collect(Collectors.toList()), page.getContinuationToken().orElse(null));
    }

    private UpdateProvisionsRequestDto toProvisionRequest(String folderId, String accountId,
                                                          UpdateProvisionsDto request) {
        UpdateProvisionsRequestDto.Builder result = UpdateProvisionsRequestDto.builder();
        result.setAccountId(accountId);
        result.setFolderId(folderId);
        request.getUpdatedProvisions().ifPresent(v -> result.setUpdatedProvisions(v.stream().map(p -> {
            ProvisionLiteDto.Builder builder = ProvisionLiteDto.builder();
            p.getResourceId().ifPresent(builder::resourceId);
            // Not nice but... provider id is unused, it is not strictly necessary anyway...
            p.getProvided().ifPresent(u -> builder.providedAmount(String.valueOf(u)));
            // Not nice but... implementation should interpret this differently for public API and front API
            p.getProvidedUnitKey().ifPresent(builder::providedAmountUnitId);
            // Old values are not supplied through public API, that's by design
            return builder.build();
        }).collect(Collectors.toList())));
        return result.build();
    }

    private UpdateProvisionOperationDto toProvisionResponse(ProvisionOperationResult response) {
        UpdateProvisionOperationDto.Builder result = UpdateProvisionOperationDto.builder();
        result.operationId(response.getOperationId());
        result.operationStatus(toStatus(response.getStatus()));
        response.getExpandedProvisionResult().ifPresent(expanded -> {
            AccountModel account = expanded.getCurrentAccount();
            String accountId = account.getId();
            String folderId = account.getFolderId();
            String providerId = account.getProviderId();
            Set<String> resourceIdsToReturn = new HashSet<>();
            Map<String, AccountsQuotasModel> updatedQuotasById = expanded.getActualUpdatedQuotas().stream()
                    .filter(q -> q.getAccountId().equals(accountId)).collect(Collectors
                            .toMap(AccountsQuotasModel::getResourceId, Function.identity()));
            Map<String, AccountsQuotasModel> originalQuotasById = expanded.getCurrentActualQuotas().stream()
                    .filter(q -> q.getAccountId().equals(accountId)).collect(Collectors
                            .toMap(AccountsQuotasModel::getResourceId, Function.identity()));
            Set<String> updatedAndOriginalResources = Sets.union(updatedQuotasById.keySet(),
                    originalQuotasById.keySet());
            updatedAndOriginalResources.forEach(resourceId -> {
                Optional<Long> updatedProvided = Optional.ofNullable(updatedQuotasById.get(resourceId))
                        .map(m -> Optional.ofNullable(m.getProvidedQuota()).orElse(0L));
                Optional<Long> originalProvided = Optional.ofNullable(originalQuotasById.get(resourceId))
                        .map(m -> Optional.ofNullable(m.getProvidedQuota()).orElse(0L));
                boolean noUpdateAndNonZero = updatedProvided.isEmpty()
                        && originalProvided.map(v -> v != 0).orElse(false);
                if (noUpdateAndNonZero || updatedProvided.isPresent()) {
                    resourceIdsToReturn.add(resourceId);
                }
            });
            resourceIdsToReturn.addAll(expanded.getRequestResourceModels().stream().map(ResourceModel::getId)
                    .collect(Collectors.toList()));
            resourceIdsToReturn.forEach(resourceId -> {
                ResourceModel resource = expanded.getAllResourceByIdMap().get(resourceId);
                UnitsEnsembleModel unitsEnsemble = expanded.getEnsembleModelByIdMap()
                        .get(resource.getUnitsEnsembleId());
                Optional<Long> updatedProvided = Optional.ofNullable(updatedQuotasById.get(resourceId))
                        .map(m -> Optional.ofNullable(m.getProvidedQuota()).orElse(0L));
                Optional<Long> originalProvided = Optional.ofNullable(originalQuotasById.get(resourceId))
                        .map(m -> Optional.ofNullable(m.getProvidedQuota()).orElse(0L));
                long providedResult = updatedProvided.or(() -> originalProvided).orElse(0L);
                long allocatedResult = updatedProvided.isPresent()
                        ? Optional.ofNullable(updatedQuotasById.get(resourceId))
                        .map(m -> Optional.ofNullable(m.getAllocatedQuota()).orElse(0L)).orElse(0L)
                        : Optional.ofNullable(originalQuotasById.get(resourceId))
                        .map(m -> Optional.ofNullable(m.getAllocatedQuota()).orElse(0L)).orElse(0L);
                Tuple2<BigDecimal, UnitModel> convertedProvided = Units
                        .convertToApi(providedResult, resource, unitsEnsemble);
                Tuple2<BigDecimal, UnitModel> convertedAllocated = Units
                        .convertToApi(allocatedResult, resource, unitsEnsemble);
                AccountProvisionDto.Builder builder = AccountProvisionDto.builder();
                builder.folderId(folderId);
                builder.accountId(accountId);
                builder.providerId(providerId);
                builder.resourceId(resourceId);
                builder.provided(convertedProvided.getT1());
                builder.providedUnitKey(convertedProvided.getT2().getKey());
                builder.allocated(convertedAllocated.getT1());
                builder.allocatedUnitKey(convertedAllocated.getT2().getKey());
                result.addProvisions(builder.build());
            });

        });
        return result.build();
    }

    private UpdateProvisionOperationStatusDto toStatus(ProvisionOperationResultStatus status) {
        switch (status) {
            case SUCCESS:
                return UpdateProvisionOperationStatusDto.SUCCESS;
            case IN_PROGRESS:
                return UpdateProvisionOperationStatusDto.IN_PROGRESS;
            default:
                throw new IllegalArgumentException("Unexpected operation status: " + status);
        }
    }

    @Schema(description = "Provisions page.")
    public static final class AccountProvisionsPageDto extends PageDto<AccountProvisionDto> {
        private AccountProvisionsPageDto(List<AccountProvisionDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }

}
