package ru.yandex.intranet.d.web.controllers.api.v1.quotas;

import java.security.Principal;
import java.util.Locale;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.services.transfer.BulkQuotaMoveService;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.quotas.QuotaTransferInputDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * Bulk quota move
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/bulkQuotaMove")
public class BulkQuotaMoveController {

    private final BulkQuotaMoveService bulkQuotaMoveService;

    public BulkQuotaMoveController(BulkQuotaMoveService bulkQuotaMoveService) {
        this.bulkQuotaMoveService = bulkQuotaMoveService;
    }


    @Operation(summary = "Transfer quota between folders.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Quota was moved."),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Referred entity not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping
    public Mono<ResponseEntity<?>> quotaMove(
            @Parameter(description = "Quotas to transfer.", required = true)
            @RequestBody QuotaTransferInputDto body,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);

        return bulkQuotaMoveService.moveQuotas(body, currentUser, locale)
                .map(result -> result.match(
                        v -> ResponseEntity.status(HttpStatus.NO_CONTENT).build(),
                        Errors::toResponse
                ));
    }

}
