package ru.yandex.intranet.d.web.controllers.api.v1.services;

import java.security.Principal;
import java.util.Locale;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.services.services.AbcServicesService;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.services.CheckServicesRequestDto;
import ru.yandex.intranet.d.web.model.services.ClosingServiceDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.AbcServiceRole;

/**
 * ABC API Controller
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@RestController
@RequestMapping("/abc")
@AbcServiceRole
public class AbcServicesController {

    private final AbcServicesService abcServicesService;

    public AbcServicesController(AbcServicesService abcServicesService) {
        this.abcServicesService = abcServicesService;
    }

    @Operation(summary = "Get list of ABC services with closing flags")
    @ApiResponses({@ApiResponse(responseCode = "200",
            description = "Requested list of ABC services with closing flags.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ClosingServiceDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/_checkClosing", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> checkClosingServices(
            @Parameter(description = "Service ids.")
            @RequestBody CheckServicesRequestDto requestDto,
            @RequestParam(value = "ignoreState", required = false, defaultValue = "false") boolean ignoreState,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ClosingServiceDto>> result = abcServicesService.checkClosing(requestDto.getServiceIds(),
                Tenants.getTenantId(currentUser), ignoreState, locale);
        return result.map(r -> r.match(Responses::okJson, Errors::toResponse));
    }
}
