package ru.yandex.intranet.d.web.controllers.api.v1.transfers;

import java.security.Principal;
import java.util.Locale;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.services.transfer.TransferRequestLogicService;
import ru.yandex.intranet.d.services.transfer.TransferRequestService;
import ru.yandex.intranet.d.services.transfer.model.ExpandedTransferRequests;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.transfers.api.CancelTransferRequestInputDto;
import ru.yandex.intranet.d.web.model.transfers.api.CreateTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferRequestVotingDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * Transfers public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/transfers")
public class ApiV1TransfersController {

    private final TransferRequestService transferRequestService;
    private final TransferRequestLogicService transferRequestLogicService;

    public ApiV1TransfersController(TransferRequestService transferRequestService,
                                    TransferRequestLogicService transferRequestLogicService) {
        this.transferRequestService = transferRequestService;
        this.transferRequestLogicService = transferRequestLogicService;
    }

    @Operation(summary = "Get one transfer request by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested transfer request.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = TransferRequestDto.class))),
            @ApiResponse(responseCode = "404", description = "'Transfer request not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Transfer request id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedTransferRequests<TransferRequestModel>>> result = transferRequestService
                .getOne(id, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(TransferDtoMapping.toTransferRequest(entity, locale)),
                Errors::toResponse));
    }

    @Operation(summary = "Create new transfer request.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Created transfer request.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = TransferRequestDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> create(
            @Parameter(description = "Transfer request to create", required = true)
            @RequestBody CreateTransferRequestDto transferRequest,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey,
            @RequestParam(value = "delayValidation", required = false, defaultValue = "false") boolean delayValidation,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedTransferRequests<TransferRequestModel>>> result = transferRequestLogicService
                .createMono(TransferDtoMapping.toCreateRequest(transferRequest), currentUser, locale, true,
                        idempotencyKey, delayValidation);
        return result.map(r -> r.match(entity -> Responses.okJson(TransferDtoMapping.toTransferRequest(entity, locale)),
                Errors::toResponse));
    }

    @Operation(summary = "Cancel existing transfer request.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Cancelled transfer request.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = TransferRequestDto.class))),
            @ApiResponse(responseCode = "404", description = "'Transfer request not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/{id}/_cancel", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> cancel(
            @Parameter(description = "Transfer request id", required = true)
            @PathVariable("id") String id,
            @RequestBody CancelTransferRequestInputDto request,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedTransferRequests<TransferRequestModel>>> result = transferRequestLogicService
                .cancelMono(id, request.getVersion().orElse(null), currentUser, locale, idempotencyKey);
        return result.map(r -> r.match(entity -> Responses.okJson(TransferDtoMapping.toTransferRequest(entity, locale)),
                Errors::toResponse));
    }

    @Operation(summary = "Vote for existing transfer request.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated transfer request.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = TransferRequestDto.class))),
            @ApiResponse(responseCode = "404", description = "'Transfer request not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/{id}/_vote", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> vote(
            @Parameter(description = "Transfer request id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Vote parameters", required = true)
            @RequestBody TransferRequestVotingDto request,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedTransferRequests<TransferRequestModel>>> result = transferRequestLogicService
                .voteMono(id, request.getVersion().orElse(null), TransferDtoMapping.toVoteParameters(request),
                        currentUser, locale, idempotencyKey);
        return result.map(r -> r.match(entity -> Responses.okJson(TransferDtoMapping.toTransferRequest(entity, locale)),
                Errors::toResponse));
    }

}
