package ru.yandex.intranet.d.web.controllers.api.v1.transfers;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.Maps;
import org.jetbrains.annotations.NotNull;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderType;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.transfers.FoldersResponsible;
import ru.yandex.intranet.d.model.transfers.LoanMeta;
import ru.yandex.intranet.d.model.transfers.LocalizedTransferApplicationErrors;
import ru.yandex.intranet.d.model.transfers.OperationStatus;
import ru.yandex.intranet.d.model.transfers.ProviderResponsible;
import ru.yandex.intranet.d.model.transfers.ProvisionTransfer;
import ru.yandex.intranet.d.model.transfers.QuotaTransfer;
import ru.yandex.intranet.d.model.transfers.ReserveResponsibleModel;
import ru.yandex.intranet.d.model.transfers.ResourceQuotaTransfer;
import ru.yandex.intranet.d.model.transfers.TransferApplicationDetails;
import ru.yandex.intranet.d.model.transfers.TransferApplicationErrors;
import ru.yandex.intranet.d.model.transfers.TransferParameters;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestType;
import ru.yandex.intranet.d.model.transfers.TransferResponsible;
import ru.yandex.intranet.d.model.transfers.TransferVotes;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.transfer.model.ExpandedTransferRequests;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestStatusDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestSubtypeDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestTypeDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestVoteTypeDto;
import ru.yandex.intranet.d.web.model.transfers.api.CreateProvisionTransferDto;
import ru.yandex.intranet.d.web.model.transfers.api.CreateQuotaResourceTransferDto;
import ru.yandex.intranet.d.web.model.transfers.api.CreateQuotaTransferDto;
import ru.yandex.intranet.d.web.model.transfers.api.CreateReserveTransferDto;
import ru.yandex.intranet.d.web.model.transfers.api.CreateTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.api.CreateTransferRequestParametersDto;
import ru.yandex.intranet.d.web.model.transfers.api.ProvisionTransferDto;
import ru.yandex.intranet.d.web.model.transfers.api.QuotaResourceTransferDto;
import ru.yandex.intranet.d.web.model.transfers.api.QuotaTransferDto;
import ru.yandex.intranet.d.web.model.transfers.api.ReserveResourceTransferDto;
import ru.yandex.intranet.d.web.model.transfers.api.ReserveTransferDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferLoanBorrowMetaDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferLoanMetaDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferLoanParametersDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferLoanPayOffMetaDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferOperationStatusDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferRequestApplicationDetailsDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferRequestApplicationFailuresDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferRequestParametersDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferRequestProviderReserveResponsibleDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferRequestProviderSuperResponsibleDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferRequestResponsibleDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferRequestResponsibleGroupDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferRequestResponsibleUserDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferRequestVoterDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferRequestVotesDto;
import ru.yandex.intranet.d.web.model.transfers.api.TransferRequestVotingDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateProvisionTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateQuotaResourceTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateQuotaTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateReserveTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateTransferRequestParametersDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferLoanBorrowParametersDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferLoanParametersDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferLoanPayOffParametersDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestVotingDto;

/**
 * Transfer DTO mapping.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class TransferDtoMapping {

    private TransferDtoMapping() {
    }

    public static FrontCreateTransferRequestDto toCreateRequest(CreateTransferRequestDto request) {
        FrontCreateTransferRequestDto.Builder result = FrontCreateTransferRequestDto.builder();
        request.getDescription().ifPresent(result::description);
        request.getAddConfirmation().ifPresent(result::addConfirmation);
        request.getRequestType().ifPresent(result::requestType);
        request.getParameters().ifPresent(v -> result.parameters(toCreateParameters(v)));
        request.getLoanParameters().ifPresent(v -> result.loanParameters(toLoanParameters(v)));
        return result.build();
    }

    public static FrontTransferRequestVotingDto toVoteParameters(TransferRequestVotingDto request) {
        return new FrontTransferRequestVotingDto(request.getVoteType().orElse(null));
    }

    public static TransferRequestDto toTransferRequest(ExpandedTransferRequests<TransferRequestModel> transfer,
                                                       Locale locale) {
        TransferRequestDto.Builder builder = TransferRequestDto.builder();
        builder.id(transfer.getTransferRequests().getId());
        builder.version(transfer.getTransferRequests().getVersion());
        transfer.getTransferRequests().getDescription().ifPresent(builder::description);
        transfer.getTransferRequests().getTrackerIssueKey().ifPresent(builder::trackerIssueKey);
        builder.requestType(TransferRequestTypeDto.fromModel(transfer.getTransferRequests().getType()));
        builder.requestSubtype(transfer.getTransferRequests().getSubtype()
                .map(TransferRequestSubtypeDto::fromModel).orElse(null));
        builder.status(TransferRequestStatusDto.fromModel(transfer.getTransferRequests().getStatus()));
        builder.createdBy(transfer.getUsers()
                .get(transfer.getTransferRequests().getCreatedBy()).getPassportUid().orElse(""));
        transfer.getTransferRequests().getUpdatedBy()
                .ifPresent(v -> builder.updatedBy(transfer.getUsers().get(v).getPassportUid().orElse("")));
        builder.createdAt(transfer.getTransferRequests().getCreatedAt());
        transfer.getTransferRequests().getUpdatedAt().ifPresent(builder::updatedAt);
        transfer.getTransferRequests().getAppliedAt().ifPresent(builder::appliedAt);
        builder.parameters(toParameters(transfer.getTransferRequests().getParameters(),
                transfer.getTransferRequests().getType(), transfer.getResources(), transfer.getUnitsEnsembles(),
                transfer.getFolders()));
        builder.transferResponsible(toResponsible(transfer.getTransferRequests().getResponsible(),
                transfer.getUsers()));
        builder.votes(toVotes(transfer.getTransferRequests().getVotes(), transfer.getUsers()));
        transfer.getTransferRequests().getApplicationDetails()
                .ifPresent(v -> builder.application(toApplication(v, locale)));
        transfer.getTransferRequests().getLoanMeta().ifPresent(v -> builder.loanMeta(toLoanMeta(v)));
        return builder.build();
    }

    private static TransferLoanMetaDto toLoanMeta(LoanMeta loanMeta) {
        TransferLoanBorrowMetaDto borrowMeta = null;
        TransferLoanPayOffMetaDto payOffMeta = null;
        switch (loanMeta.getOperationType()) {
            case BORROW -> borrowMeta = new TransferLoanBorrowMetaDto(
                    Objects.requireNonNull(loanMeta.getBorrowDueDate()).getLocalDate(),
                    new ArrayList<>(Objects.requireNonNullElse(loanMeta.getBorrowLoanIds(), Set.of())));
            case PAY_OFF -> payOffMeta = new TransferLoanPayOffMetaDto(Objects.requireNonNull(loanMeta
                    .getPayOffLoanId()));
            default -> {
            }
        }
        return new TransferLoanMetaDto(borrowMeta, payOffMeta, loanMeta.getProvideOverCommitReserve());
    }

    private static FrontCreateTransferRequestParametersDto toCreateParameters(
            CreateTransferRequestParametersDto parameters) {
        FrontCreateTransferRequestParametersDto.Builder result = FrontCreateTransferRequestParametersDto.builder();
        parameters.getQuotaTransfers().ifPresent(v -> v.forEach(transfer ->
                result.addQuotaTransfer(toCreateQuotaTransfer(transfer))));
        parameters.getReserveTransfer().ifPresent(v -> result.reserveTransfer(toCreateReserveTransfer(v)));
        parameters.getProvisionTransfers().ifPresent(v ->
                v.forEach(t -> result.addProvisionTransfer(toCreateProvisionTransfer(t))));
        return result.build();
    }

    private static FrontTransferLoanParametersDto toLoanParameters(TransferLoanParametersDto loanParameters) {
        FrontTransferLoanBorrowParametersDto borrowParameters = null;
        FrontTransferLoanPayOffParametersDto payOffParameters = null;
        if (loanParameters.getBorrowParameters() != null) {
            borrowParameters = new FrontTransferLoanBorrowParametersDto(loanParameters.getBorrowParameters()
                    .getDueDate());
        }
        if (loanParameters.getPayOffParameters() != null) {
            payOffParameters = new FrontTransferLoanPayOffParametersDto(loanParameters.getPayOffParameters()
                    .getLoanId());
        }
        return new FrontTransferLoanParametersDto(borrowParameters, payOffParameters,
                loanParameters.getProvideOverCommitReserve());
    }

    private static FrontCreateQuotaTransferDto toCreateQuotaTransfer(CreateQuotaTransferDto transfer) {
        FrontCreateQuotaTransferDto.Builder result = FrontCreateQuotaTransferDto.builder();
        transfer.getFolderId().ifPresent(result::destinationFolderId);
        // Not nice but... service id is not present here, validator should behave differently depending on endpoint
        transfer.getResourceTransfers().ifPresent(v -> v.forEach(resourceTransfer ->
                result.addResourceTransfer(toQuotaResourceTransfer(resourceTransfer))));
        return result.build();
    }

    @NotNull
    private static FrontCreateQuotaResourceTransferDto toQuotaResourceTransfer(
            CreateQuotaResourceTransferDto resourceTransfer
    ) {
        FrontCreateQuotaResourceTransferDto.Builder builder = FrontCreateQuotaResourceTransferDto.builder();
        resourceTransfer.getResourceId().ifPresent(builder::resourceId);
        // Not nice but... incoming provider id is ignored here, it isn't strictly necessary anyway...
        resourceTransfer.getDelta().ifPresent(l -> builder.delta(String.valueOf(l)));
        // Not nice but... this should be interpreted differently by validator depending on endpoint
        resourceTransfer.getDeltaUnitKey().ifPresent(builder::deltaUnitId);
        return builder.build();
    }

    private static FrontCreateReserveTransferDto toCreateReserveTransfer(CreateReserveTransferDto transfer) {
        FrontCreateReserveTransferDto.Builder result = FrontCreateReserveTransferDto.builder();
        transfer.getProviderId().ifPresent(result::providerId);
        transfer.getFolderId().ifPresent(result::destinationFolderId);
        // Not nice but... service id is not present here, validator should behave differently depending on endpoint
        transfer.getResourceTransfers().ifPresent(v -> v.forEach(resourceTransfer -> {
            FrontCreateQuotaResourceTransferDto.Builder builder = FrontCreateQuotaResourceTransferDto.builder();
            resourceTransfer.getResourceId().ifPresent(builder::resourceId);
            resourceTransfer.getDelta().ifPresent(l -> builder.delta(String.valueOf(l)));
            // Not nice but... this should be interpreted differently by validator depending on endpoint
            resourceTransfer.getDeltaUnitKey().ifPresent(builder::deltaUnitId);
            result.addResourceTransfer(builder.build());
        }));
        return result.build();
    }

    private static FrontCreateProvisionTransferDto toCreateProvisionTransfer(CreateProvisionTransferDto transfer) {
        return new FrontCreateProvisionTransferDto(
                transfer.getSourceAccountId(),
                transfer.getSourceFolderId(),
                null,
                transfer.getDestinationAccountId(),
                transfer.getDestinationFolderId(),
                null,
                toQuotaResourceTransfers(transfer.getSourceAccountTransfers()),
                toQuotaResourceTransfers(transfer.getDestinationAccountTransfers())
        );
    }

    private static List<FrontCreateQuotaResourceTransferDto> toQuotaResourceTransfers(
            @Nullable Collection<CreateQuotaResourceTransferDto> transfers
    ) {
        if (transfers == null) {
            return List.of();
        }
        return transfers.stream().map(TransferDtoMapping::toQuotaResourceTransfer).collect(Collectors.toList());
    }

    private static TransferRequestParametersDto toParameters(TransferParameters parameters,
                                                             TransferRequestType type,
                                                             Map<String, ResourceModel> resources,
                                                             Map<String, UnitsEnsembleModel> unitsEnsembles,
                                                             Map<String, FolderModel> folders) {
        TransferRequestParametersDto.Builder result = TransferRequestParametersDto.builder();
        switch (type) {
            case QUOTA_TRANSFER ->
                    result.addQuotaTransfers(toQuotaTransferParameters(parameters.getQuotaTransfers(),
                    resources, unitsEnsembles));
            case RESERVE_TRANSFER ->
                    result.reserveTransfer(toReserveTransferParameters(parameters.getQuotaTransfers(),
                    folders, resources, unitsEnsembles));
            case PROVISION_TRANSFER ->
                    result.addProvisionTransfers(toProvisionTransferParameters(parameters.getProvisionTransfers(),
                    resources, unitsEnsembles));
            default -> throw new IllegalArgumentException("Unsupported transfer type: " + type);
        }
        return result.build();
    }

    private static List<ProvisionTransferDto> toProvisionTransferParameters(
            Set<ProvisionTransfer> provisionTransfers,
            Map<String, ResourceModel> resources,
            Map<String, UnitsEnsembleModel> unitsEnsembles
    ) {
        return provisionTransfers.stream()
                .map(p -> toProvisionTransfer(p, resources, unitsEnsembles))
                .collect(Collectors.toList());
    }

    private static ProvisionTransferDto toProvisionTransfer(ProvisionTransfer provisionTransfer,
                                                            Map<String, ResourceModel> resources,
                                                            Map<String, UnitsEnsembleModel> unitsEnsembles) {
        return new ProvisionTransferDto(
            provisionTransfer.getSourceAccountId(),
            provisionTransfer.getDestinationAccountId(),
            provisionTransfer.getSourceFolderId(),
            provisionTransfer.getDestinationFolderId(),
            Long.toString(provisionTransfer.getSourceServiceId()),
            Long.toString(provisionTransfer.getDestinationServiceId()),
            toResourceTransfers(provisionTransfer.getSourceAccountTransfers(), resources, unitsEnsembles),
            toResourceTransfers(provisionTransfer.getDestinationAccountTransfers(), resources, unitsEnsembles),
            provisionTransfer.getOperationId()
        );
    }

    private static List<QuotaTransferDto> toQuotaTransferParameters(
            Set<QuotaTransfer> transfers,
            Map<String, ResourceModel> resources, Map<String, UnitsEnsembleModel> unitsEnsembles) {
        return transfers.stream().map(transfer -> toQuotaTransfer(transfer, resources, unitsEnsembles))
                .collect(Collectors.toList());
    }

    private static QuotaTransferDto toQuotaTransfer(QuotaTransfer transfer,
                                                    Map<String, ResourceModel> resources,
                                                    Map<String, UnitsEnsembleModel> unitsEnsembles) {
        QuotaTransferDto.Builder result = QuotaTransferDto.builder();
        result.folderId(transfer.getDestinationFolderId());
        transfer.getTransfers().forEach(resourceTransfer -> result
                .addResourceTransfer(toResourceTransfer(resourceTransfer, resources, unitsEnsembles)));
        return result.build();
    }

    private static List<QuotaResourceTransferDto> toResourceTransfers(
            Collection<? extends ResourceQuotaTransfer> transfers,
            Map<String, ResourceModel> resources,
            Map<String, UnitsEnsembleModel> unitsEnsembles
    ) {
        return transfers.stream()
                .map(t -> toResourceTransfer(t, resources, unitsEnsembles))
                .collect(Collectors.toList());
    }

    private static QuotaResourceTransferDto toResourceTransfer(ResourceQuotaTransfer transfer,
                                                               Map<String, ResourceModel> resources,
                                                               Map<String, UnitsEnsembleModel> unitsEnsembles) {
        ResourceModel resource = resources.get(transfer.getResourceId());
        UnitsEnsembleModel unitsEnsemble = unitsEnsembles.get(resource.getUnitsEnsembleId());
        Tuple2<BigDecimal, UnitModel> convertedDelta = Units.convertToApi(transfer.getDelta(), resource, unitsEnsemble);
        QuotaResourceTransferDto.Builder result = QuotaResourceTransferDto.builder();
        result.resourceId(transfer.getResourceId());
        result.providerId(resource.getProviderId());
        result.delta(convertedDelta.getT1().longValueExact());
        result.deltaUnitKey(convertedDelta.getT2().getKey());
        return result.build();
    }

    private static ReserveTransferDto toReserveTransferParameters(Set<QuotaTransfer> transfers,
                                                                  Map<String, FolderModel> folders,
                                                                  Map<String, ResourceModel> resources,
                                                                  Map<String, UnitsEnsembleModel> unitsEnsembles) {
        ReserveTransferDto.Builder result = ReserveTransferDto.builder();
        String destinationFolderId = getDestinationFolderId(transfers, folders);
        String sourceProviderId = getSourceProviderId(transfers, resources);
        result.folderId(destinationFolderId);
        result.providerId(sourceProviderId);
        transfers.stream().filter(t -> t.getDestinationFolderId().equals(destinationFolderId))
                .forEach(t -> t.getTransfers().forEach(r -> result.addResourceTransfer(toReserveResourceTransfer(r,
                        resources, unitsEnsembles))));
        return result.build();
    }

    private static ReserveResourceTransferDto toReserveResourceTransfer(
            ResourceQuotaTransfer transfer, Map<String, ResourceModel> resources,
            Map<String, UnitsEnsembleModel> unitsEnsembles) {
        ResourceModel resource = resources.get(transfer.getResourceId());
        UnitsEnsembleModel unitsEnsemble = unitsEnsembles.get(resource.getUnitsEnsembleId());
        Tuple2<BigDecimal, UnitModel> convertedDelta = Units.convertToApi(transfer.getDelta(), resource, unitsEnsemble);
        ReserveResourceTransferDto.Builder result = ReserveResourceTransferDto.builder();
        result.resourceId(transfer.getResourceId());
        result.delta(convertedDelta.getT1().longValueExact());
        result.deltaUnitKey(convertedDelta.getT2().getKey());
        return result.build();
    }

    private static String getDestinationFolderId(Set<QuotaTransfer> transfers, Map<String, FolderModel> folders) {
        return transfers.stream().filter(transfer -> folders
                        .get(transfer.getDestinationFolderId()).getFolderType() != FolderType.PROVIDER_RESERVE)
                .findFirst().map(ru.yandex.intranet.d.model.transfers.QuotaTransfer::getDestinationFolderId)
                .orElseThrow();
    }

    private static String getSourceProviderId(Set<QuotaTransfer> transfers, Map<String, ResourceModel> resources) {
        Set<String> providerIds = new HashSet<>();
        transfers.forEach(t -> t.getTransfers()
                .forEach(r -> providerIds.add(resources.get(r.getResourceId()).getProviderId())));
        return providerIds.stream().findFirst().orElseThrow();
    }

    private static TransferRequestResponsibleDto toResponsible(TransferResponsible responsible,
                                                               Map<String, UserModel> users) {
        TransferRequestResponsibleDto.Builder result = TransferRequestResponsibleDto.builder();
        responsible.getResponsible().forEach(r -> result.addGroup(toGroupedTransferResponsible(r, users)));
        responsible.getProviderResponsible()
                .forEach(r -> result.addProviderSuperResponsibleUser(toProviderSuperResponsible(r, users)));
        responsible.getReserveResponsibleModel()
                .ifPresent(r -> result.addProviderReserveResponsibleUser(toProviderReserveResponsible(r, users)));
        return result.build();
    }

    private static TransferRequestResponsibleGroupDto toGroupedTransferResponsible(
            FoldersResponsible foldersResponsible, Map<String, UserModel> users) {
        TransferRequestResponsibleGroupDto.Builder result = TransferRequestResponsibleGroupDto.builder();
        result.addFolderIds(foldersResponsible.getFolderIds());
        Map<String, Set<Long>> userServices = new HashMap<>();
        foldersResponsible.getResponsible().forEach(serviceResponsible -> serviceResponsible.getResponsibleIds()
                .forEach(userId -> userServices.computeIfAbsent(userId, k -> new HashSet<>())
                        .add(serviceResponsible.getServiceId())));
        userServices.forEach((userId, serviceIds) -> {
            TransferRequestResponsibleUserDto.Builder responsibleBuilder = TransferRequestResponsibleUserDto.builder();
            responsibleBuilder.responsible(users.get(userId).getPassportUid().orElse(""));
            responsibleBuilder.addServiceIds(serviceIds);
            result.addResponsibleUser(responsibleBuilder.build());
        });
        return result.build();
    }

    private static TransferRequestProviderSuperResponsibleDto toProviderSuperResponsible(
            ProviderResponsible responsible, Map<String, UserModel> users) {
        TransferRequestProviderSuperResponsibleDto.Builder result
                = TransferRequestProviderSuperResponsibleDto.builder();
        result.responsible(users.get(responsible.getResponsibleId()).getPassportUid().orElse(""));
        result.addProviderIds(responsible.getProviderIds());
        return result.build();
    }

    private static TransferRequestProviderReserveResponsibleDto toProviderReserveResponsible(
            ReserveResponsibleModel responsible, Map<String, UserModel> users) {
        TransferRequestProviderReserveResponsibleDto.Builder result
                = TransferRequestProviderReserveResponsibleDto.builder();
        result.addResponsibleUsers(responsible.getResponsibleIds().stream()
                .map(v -> users.get(v).getPassportUid().orElse("")).collect(Collectors.toList()));
        result.providerId(responsible.getProviderId());
        result.folderId(responsible.getFolderId());
        result.serviceId(responsible.getServiceId());
        return result.build();
    }

    private static TransferRequestVotesDto toVotes(TransferVotes votes,
                                                   Map<String, UserModel> users) {
        TransferRequestVotesDto.Builder result = TransferRequestVotesDto.builder();
        votes.getVotes().forEach(vote -> {
            TransferRequestVoterDto.Builder voter = TransferRequestVoterDto.builder();
            voter.voter(users.get(vote.getUserId()).getPassportUid().orElse(""));
            voter.addFolderIds(vote.getFolderIds());
            voter.addServiceIds(vote.getServiceIds());
            voter.timestamp(vote.getTimestamp());
            voter.voteType(TransferRequestVoteTypeDto.fromModel(vote.getType()));
            voter.addProviderIds(vote.getProviderIds());
            result.addVoters(voter.build());
        });
        return result.build();
    }

    private static TransferRequestApplicationDetailsDto toApplication(TransferApplicationDetails applicationDetails,
                                                                      Locale locale) {
        TransferRequestApplicationDetailsDto.Builder result = TransferRequestApplicationDetailsDto.builder();
        Optional<TransferApplicationErrors> errorsO = Locales.selectObject(applicationDetails.getErrorsEn(),
                applicationDetails.getErrorsRu(), locale);
        errorsO.ifPresent(errors -> result.failures(toFailures(errors)));
        result.operationStatusById(Maps.transformValues(applicationDetails.getOperationStatusById(),
                TransferDtoMapping::toOperationStatus));
        result.failuresByOperationId(Maps.transformValues(applicationDetails.getErrorsByOperationId(),
                e -> toFailures(e, locale)));
        return result.build();
    }

    @NotNull
    private static TransferRequestApplicationFailuresDto toFailures(LocalizedTransferApplicationErrors errors,
                                                                    Locale locale) {
        TransferApplicationErrors transferApplicationErrors = Locales.selectObject(errors.getErrorsEn(),
                errors.getErrorsRu(), locale);
        return toFailures(transferApplicationErrors);
    }

    @NotNull
    private static TransferRequestApplicationFailuresDto toFailures(TransferApplicationErrors errors) {
        TransferRequestApplicationFailuresDto.Builder failures = TransferRequestApplicationFailuresDto.builder();
        failures.addErrors(errors.getErrors());
        errors.getFieldErrors().forEach(failures::addFieldErrors);
        failures.addDetails(errors.getDetails());
        return failures.build();
    }

    private static TransferOperationStatusDto toOperationStatus(OperationStatus operationStatus) {
        return switch (operationStatus) {
            case COMPLETED -> TransferOperationStatusDto.COMPLETED;
            case FAILED -> TransferOperationStatusDto.FAILED;
            case EXECUTING -> TransferOperationStatusDto.EXECUTING;
            case UNKNOWN -> TransferOperationStatusDto.UNKNOWN;
        };
    }
}
