package ru.yandex.intranet.d.web.controllers.api.v1.units;

import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.units.GrammaticalCase;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.units.UnitsComparator;
import ru.yandex.intranet.d.services.units.UnitsEnsemblesService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.units.UnitDto;
import ru.yandex.intranet.d.web.model.units.UnitsEnsembleDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * Units ensembles public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/api/v1/unitsEnsembles")
public class ApiV1UnitsEnsemblesController {

    private final UnitsEnsemblesService unitsEnsemblesService;

    public ApiV1UnitsEnsemblesController(UnitsEnsemblesService unitsEnsemblesService) {
        this.unitsEnsemblesService = unitsEnsemblesService;
    }

    @Operation(summary = "Get one units ensemble by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested units ensemble.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = UnitsEnsembleDto.class))),
            @ApiResponse(responseCode = "404", description = "'Units ensemble not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Units ensemble id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<UnitsEnsembleModel>> result = unitsEnsemblesService.getById(id, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toUnitsEnsemble(entity, locale)),
                Errors::toResponse));
    }

    @Operation(summary = "Get one unit by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested unit.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = UnitsEnsembleDto.class))),
            @ApiResponse(responseCode = "404", description = "'Unit not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{ensembleId}/units/{unitId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOneUnit(
            @Parameter(description = "Units ensemble id", required = true)
            @PathVariable("ensembleId") String ensembleId,
            @Parameter(description = "Unit id", required = true)
            @PathVariable("unitId") String unitId,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<UnitModel>> result = unitsEnsemblesService.getUnitById(ensembleId, unitId, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(toUnit(entity, locale)),
                Errors::toResponse));
    }

    @Operation(summary = "Get one units ensembles page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested units ensembles page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = UnitsEnsemblePageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPage(
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Page<UnitsEnsembleModel>>> result = unitsEnsemblesService
                .getPage(new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p, locale)), Errors::toResponse));
    }

    private PageDto<UnitsEnsembleDto> toPage(Page<UnitsEnsembleModel> p, Locale locale) {
        return new PageDto<>(p.getItems().stream().map(m -> toUnitsEnsemble(m, locale))
                .collect(Collectors.toList()), p.getContinuationToken().orElse(null));
    }

    private UnitsEnsembleDto toUnitsEnsemble(UnitsEnsembleModel model, Locale locale) {
        return new UnitsEnsembleDto(
                model.getId(), // id
                model.getVersion(), // version
                Locales.select(model.getNameEn(), model.getNameRu(), locale), // name
                Locales.select(model.getDescriptionEn(), model.getDescriptionRu(), locale), // description
                model.isFractionsAllowed(), // fractionsAllowed
                model.getUnits().stream().sorted(UnitsComparator.INSTANCE)
                        .map(u -> toUnit(u, locale)).collect(Collectors.toList()), // units
                model.getKey() // key
        );
    }

    private UnitDto toUnit(UnitModel model, Locale locale) {
        return new UnitDto(
                model.getId(), // id
                model.getKey(), // key
                Locales.select(model.getShortNameSingularEn(),
                        model.getShortNameSingularRu().get(GrammaticalCase.NOMINATIVE), locale), // shortName
                Locales.select(model.getLongNameSingularEn(),
                        model.getLongNameSingularRu().get(GrammaticalCase.NOMINATIVE), locale), // longName
                model.getBase(), // base
                model.getPower() // power
        );
    }

    @Schema(description = "Units ensembles page.")
    private static final class UnitsEnsemblePageDto extends PageDto<UnitsEnsembleDto> {
        private UnitsEnsemblePageDto(List<UnitsEnsembleDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }

}
