package ru.yandex.intranet.d.web.controllers.front;

import java.math.BigDecimal;
import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.accounts.AccountLogicService;
import ru.yandex.intranet.d.services.accounts.AccountService;
import ru.yandex.intranet.d.services.accounts.AccountsReadService;
import ru.yandex.intranet.d.services.accounts.ReserveAccountsService;
import ru.yandex.intranet.d.services.accounts.model.AccountOperationResult;
import ru.yandex.intranet.d.services.accounts.model.AccountOperationResultStatus;
import ru.yandex.intranet.d.services.accounts.model.AccountsByFolderResult;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyResponseDto;
import ru.yandex.intranet.d.services.quotas.QuotasHelper;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.AccountDto;
import ru.yandex.intranet.d.web.model.AvailableResourcesDto;
import ru.yandex.intranet.d.web.model.CreateAccountExpandedAnswerDto;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.accounts.AccountWithQuotaDto;
import ru.yandex.intranet.d.web.model.accounts.AccountsWithQuotaDto;
import ru.yandex.intranet.d.web.model.folders.FrontAccountInputDto;
import ru.yandex.intranet.d.web.model.folders.FrontAccountOperationDto;
import ru.yandex.intranet.d.web.model.folders.FrontPutAccountDto;
import ru.yandex.intranet.d.web.model.folders.FrontReserveAccountsDto;
import ru.yandex.intranet.d.web.model.folders.front.ExpandedAccountResource;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserRole;

import static ru.yandex.intranet.d.services.quotas.QuotasHelper.getAmountDto;
import static ru.yandex.intranet.d.services.quotas.QuotasHelper.toProvidedAndNotAllocated;

/**
 * Front accounts controller
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
@UserRole
@RestController
@RequestMapping("front/accounts")
public class FrontAccountsController {

    private final AccountService accountService;
    private final AccountsReadService accountsReadService;
    private final AccountLogicService accountLogicService;
    private final ReserveAccountsService reserveAccountsService;

    public FrontAccountsController(AccountService accountService, AccountsReadService accountsReadService,
                                   AccountLogicService accountLogicService,
                                   ReserveAccountsService reserveAccountsService) {
        this.accountService = accountService;
        this.accountsReadService = accountsReadService;
        this.accountLogicService = accountLogicService;
        this.reserveAccountsService = reserveAccountsService;
    }


    @Operation(description = "Создать новый аккаунт. ",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Информация о созданном аккаунте",
                            content = @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = AccountDto.class)
                            )),
                    @ApiResponse(responseCode = "422", description = "Данные не прошли валидацию, ошибку смотрите в " +
                            "fieldErrors (ключом будет имя поля с ошибкой).",
                            content = @Content(
                                    mediaType = "application/json",
                                    schema = @Schema(implementation = ErrorCollectionDto.class)
                            ))
            })
    @PostMapping(consumes = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> createAccount(
            Principal principal,
            Locale locale,
            @Parameter(description = "New account.", required = true)
            @RequestBody FrontAccountInputDto accountInputDto,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        return accountLogicService.createAccountMono(accountInputDto, currentUser, locale, idempotencyKey)
                .map(result -> result.match(
                        answer -> ResponseEntity.ok().contentType(MediaType.APPLICATION_JSON)
                                .body(answer.getExpandedProvider().getAccounts().get(0).getAccount()),
                        Errors::toResponse
                ));
    }

    @Operation(summary = "Update account.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated account.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FrontAccountOperationDto.class))),
            @ApiResponse(responseCode = "202", description = "Update in progress.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = FrontAccountOperationDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PutMapping(value = "/{accountId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> putAccount(
            @Parameter(description = "Account id.", required = true)
            @PathVariable String accountId,
            @Parameter(description = "Updated account.", required = true)
            @RequestBody FrontPutAccountDto accountPutDto,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey,
            Principal principal,
            Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<AccountOperationResult>> result = accountLogicService
                .putAccountMono(accountId, accountPutDto, currentUser, locale, idempotencyKey);
        return result.map(r -> r.match(entity -> {
            if (entity.getOperationStatus() == AccountOperationResultStatus.SUCCESS) {
                return Responses.okJson(toAccountOperation(entity));
            } else {
                return Responses.acceptedJson(toAccountOperation(entity));
            }
        }, Errors::toResponse));
    }

    @Operation(description = "Создать новый аккаунт.",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Информация о созданном аккаунте " +
                            "и квотам по умолчанию",
                            content = @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = CreateAccountExpandedAnswerDto.class)
                            )),
                    @ApiResponse(responseCode = "422", description = "Данные не прошли валидацию, ошибку смотрите в " +
                            "fieldErrors (ключом будет имя поля с ошибкой).",
                            content = @Content(
                                    mediaType = "application/json",
                                    schema = @Schema(implementation = ErrorCollectionDto.class)
                            ))
            })
    @PostMapping(value = "/_expanded", consumes = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> createAccountExpanded(
            Principal principal,
            Locale locale,
            @Parameter(description = "New account.", required = true)
            @RequestBody FrontAccountInputDto accountInputDto,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        return accountLogicService.createAccountMono(accountInputDto, currentUser, locale, idempotencyKey)
                .map(result -> result.match(
                        (CreateAccountExpandedAnswerDto answer) -> ResponseEntity.ok()
                                .contentType(MediaType.APPLICATION_JSON)
                                .body(answer),
                        Errors::toResponse
                ));
    }

    @Operation(description = "Получение доступных для аккаунта ресурсов.",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Информация о доступных ресурсах для аккаунта",
                            content = @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = AvailableResourcesDto.class)
                            )),
                    @ApiResponse(responseCode = "404", description = "Аккаунт не найден.",
                            content = @Content(
                                    mediaType = "application/json",
                                    schema = @Schema(implementation = ErrorCollectionDto.class)
                            ))
            })
    @GetMapping("/{accountId}/available-resources")
    public Mono<ResponseEntity<?>> getAccountAvailableResources(
            @Parameter(description = "Account id.", required = true)
            @PathVariable String accountId,
            Principal principal,
            Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        return accountService.getAvailableResources(accountId, currentUser, locale)
                .map(result -> result.match(resources -> Responses.okJson(new AvailableResourcesDto(resources, locale)),
                        Errors::toResponse
                ));
    }

    @Operation(description = "Получение аккаунтов фолдера.",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Аккаунты с квотами",
                            content = @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = AccountWithQuotaDto.class)
                            )),
                    @ApiResponse(responseCode = "404", description = "Фолдер или провайдер не найден.",
                            content = @Content(
                                    mediaType = "application/json",
                                    schema = @Schema(implementation = ErrorCollectionDto.class)
                            ))
            })
    @GetMapping("/{folderId}/{providerId}")
    public Mono<ResponseEntity<?>> getAccountsByFolder(
            @Parameter(description = "Folder id.", required = true)
            @PathVariable String folderId,
            @Parameter(description = "Provider id.", required = true)
            @PathVariable String providerId,
            Principal principal,
            Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        return accountsReadService.getAccountsByFolder(folderId, providerId, currentUser, locale)
                .map(result -> result.match(accounts -> Responses.okJson(toDto(accounts, locale)),
                        Errors::toResponse
                ));
    }

    @Operation(summary = "Get reserve accounts.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Reserve accounts.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FrontReserveAccountsDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/_reserve", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getReserveAccounts(
            @Parameter(description = "Service id to search reserve accounts for")
            @RequestParam(value = "serviceId", required = false) Long serviceId,
            @Parameter(description = "Provider id to search reserve accounts for")
            @RequestParam(value = "providerId", required = false) String providerId,
            Principal principal,
            Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        return reserveAccountsService.findReserveAccountsMono(serviceId, providerId, currentUser, locale)
                .map(result -> result.match(Responses::okJson, Errors::toResponse));
    }

    private AccountsWithQuotaDto toDto(AccountsByFolderResult accounts, Locale locale) {
        AccountsWithQuotaDto.Builder accountsBuilder = AccountsWithQuotaDto.builder();

        Map<String, AccountSpaceModel> accountSpaceById = accounts.getAccountSpaceModels().stream()
                .collect(Collectors.toMap(AccountSpaceModel::getId, Function.identity()));

        Map<String, List<AccountsQuotasModel>> quotasByAccountId = accounts.getAccountsQuotasModels()
                .stream()
                .collect(Collectors.groupingBy(AccountsQuotasModel::getAccountId));

        Map<String, ResourceSegmentationModel> segmentationById = accounts.getSegmentations().stream()
                .collect(Collectors.toMap(ResourceSegmentationModel::getId, Function.identity()));

        Map<String, ResourceSegmentModel> segmentById = accounts.getSegments().stream()
                .collect(Collectors.toMap(ResourceSegmentModel::getId, Function.identity()));

        Map<String, ResourceModel> resourcesById = accounts.getResources().stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));

        Map<String, UnitsEnsembleModel> unitEnsemblesById = accounts.getUnitEnsembles().stream()
                .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));

        accounts.getAccountModels().forEach(accountModel -> {
            AccountWithQuotaDto.Builder builder = AccountWithQuotaDto.builder();
            Optional<String> accountsSpacesId = accountModel.getAccountsSpacesId();
            builder.id(accountModel.getId())
                    .displayName(accountModel.getDisplayName().orElse(null))
                    .provisions(toProvisionsDto(quotasByAccountId.getOrDefault(accountModel.getId(), List.of()),
                            resourcesById, unitEnsemblesById, locale))
                    .accountsSpaceId(accountsSpacesId.orElse(null));

            accountsSpacesId.ifPresent(s -> builder.accountsSpaceKey(
                    toAccountSpaceKey(accountSpaceById.get(s), segmentationById, segmentById)));


            accountsBuilder.accountWithQuotaDto(builder.build());
        });

        return accountsBuilder.build();
    }

    private List<ExpandedAccountResource> toProvisionsDto(List<AccountsQuotasModel> accountsQuotasModels, Map<String,
            ResourceModel> resourcesById, Map<String, UnitsEnsembleModel> unitEnsemblesById, Locale locale) {
        return accountsQuotasModels.stream()
                .filter(accountsQuotasModel -> accountsQuotasModel.getProvidedQuota() != 0
                        || accountsQuotasModel.getAllocatedQuota() != 0)
                .map(accountsQuota -> {
                    String resourceId = accountsQuota.getResourceId();
                    ResourceModel resourceModel = resourcesById.get(resourceId);
                    return new ExpandedAccountResource(resourceId,
                            getAmountDto(QuotasHelper.toBigDecimal(accountsQuota.getProvidedQuota()), resourceModel,
                                    unitEnsemblesById.get(resourceModel.getUnitsEnsembleId()), locale), BigDecimal.ZERO,
                            getAmountDto(QuotasHelper.toBigDecimal(accountsQuota.getAllocatedQuota()), resourceModel,
                                    unitEnsemblesById.get(resourceModel.getUnitsEnsembleId()), locale), BigDecimal.ZERO,
                            getAmountDto(toProvidedAndNotAllocated(accountsQuota.getProvidedQuota(),
                                            accountsQuota.getAllocatedQuota()), resourceModel,
                                    unitEnsemblesById.get(resourceModel.getUnitsEnsembleId()), locale
                            )
                    );
                }).collect(Collectors.toList());
    }

    private AccountsSpaceKeyResponseDto toAccountSpaceKey(AccountSpaceModel accountSpaceModel, Map<String,
            ResourceSegmentationModel> segmentationById, Map<String, ResourceSegmentModel> segmentById) {
        List<SegmentKeyResponseDto> segmentKeyResponseList = accountSpaceModel.getSegments().stream()
                .map(segment -> new SegmentKeyResponseDto(segmentationById.get(segment.getSegmentationId()).getKey(),
                        segmentById.get(segment.getSegmentId()).getKey()))
                .collect(Collectors.toList());
        return new AccountsSpaceKeyResponseDto(segmentKeyResponseList);
    }

    private FrontAccountOperationDto toAccountOperation(AccountOperationResult operation) {
        return new FrontAccountOperationDto(AccountOperationResultStatus.toDto(operation.getOperationStatus()),
                operation.getAccount().map(AccountDto::fromModel).orElse(null), operation.getOperationId());
    }

}
