package ru.yandex.intranet.d.web.controllers.front;

import java.security.Principal;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nullable;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.services.resources.AccountsSpacesService;
import ru.yandex.intranet.d.services.resources.AccountsSpacesUtils;
import ru.yandex.intranet.d.services.resources.ExpandedAccountsSpaces;
import ru.yandex.intranet.d.services.resources.SelectionTreeNode;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.resources.AccountsSpaceDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * FrontAccountsSpacesController
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/front/providers/{providerId}/accountsSpaces")
public class FrontAccountsSpacesController {
    private final AccountsSpacesService accountsSpacesService;
    private final AccountsSpacesUtils accountsSpacesUtils;

    public FrontAccountsSpacesController(AccountsSpacesService accountsSpacesService,
                                         AccountsSpacesUtils accountsSpacesUtils) {
        this.accountsSpacesService = accountsSpacesService;
        this.accountsSpacesUtils = accountsSpacesUtils;
    }

    @Operation(summary = "Get one accounts spaces page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested accounts spaces page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = AccountsSpacesPageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPage(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedAccountsSpaces<Page<AccountSpaceModel>>>> result = accountsSpacesService
                .getPageExpanded(providerId, new PageRequest(pageToken, limit), currentUser, locale, false);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p, locale)),
                Errors::toResponse));
    }

    @Operation(summary = "Get accounts spaces selection tree.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested accounts spaces as selection tree.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = SelectionTreeNodeDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/_selectionTree", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getSelectionTree(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("providerId") String providerId,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedAccountsSpaces<SelectionTreeNode>>> result =
                accountsSpacesService.getSelectionTree(providerId, currentUser, locale);
        return result.map(r -> r.match(t -> Responses.okJson(toDto(t, locale)), Errors::toResponse));
    }

    private SelectionTreeNodeDto toDto(
            ExpandedAccountsSpaces<SelectionTreeNode> node,
            Locale locale
    ) {
        return toDto(node.getAccountsSpaces(), node.getSegmentations(), node.getSegments(), null, locale);
    }

    private SelectionTreeNodeDto toDto(
            SelectionTreeNode root,
            Map<String, ResourceSegmentationModel> resourceSegmentations,
            Map<String, ResourceSegmentModel> resourceSegments,
            ResourceSegmentModel currentNodeSegment,
            Locale locale
    ) {
        if (root.getValue() != null) {
            return new SelectionTreeNodeDto(
                    root.getSegmentationName(),
                    null,
                    accountsSpacesUtils.toDto(root.getValue(), resourceSegmentations, resourceSegments, locale),
                    currentNodeSegment != null ? currentNodeSegment.getUncommon().orElse(false) : null
            );
        }
        if (root.getChildrenBySegmentId() == null) {
            return new SelectionTreeNodeDto(root.getSegmentationName(), null, null,
                    currentNodeSegment != null ? currentNodeSegment.getUncommon().orElse(false) : null);
        }
        Map<String, SelectionTreeNodeDto> childrenBySegmentName = new HashMap<>(root.getChildrenBySegmentId().size());
        root.getChildrenBySegmentId().forEach((segmentId, node) -> childrenBySegmentName.put(
                getName(resourceSegments.get(segmentId), locale),
                toDto(node, resourceSegmentations, resourceSegments, resourceSegments.get(segmentId), locale)
        ));
        return new SelectionTreeNodeDto(root.getSegmentationName(), childrenBySegmentName, null,
                currentNodeSegment != null ? currentNodeSegment.getUncommon().orElse(false) : null);
    }

    private String getName(ResourceSegmentModel model, Locale locale) {
        return Locales.select(model.getNameEn(), model.getNameRu(), locale);
    }

    private AccountsSpacesPageDto toPage(
            ExpandedAccountsSpaces<Page<AccountSpaceModel>> em, Locale locale) {
        return new AccountsSpacesPageDto(
                accountsSpacesUtils.toDtosFromPage(em, locale),
                em.getAccountsSpaces().getContinuationToken().orElse(null)
        );
    }

    @Schema(description = "Accounts spaces page.")
    public static class AccountsSpacesPageDto extends PageDto<AccountsSpaceDto> {
        public AccountsSpacesPageDto(List<AccountsSpaceDto> items, String nextPageToken) {
            super(items, nextPageToken);
        }
    }

    @Schema(description = "Accounts selection tree node.")
    public static final class SelectionTreeNodeDto {
        private final String segmentationName;
        private final Map<String, SelectionTreeNodeDto> childrenBySegmentName;
        private final AccountsSpaceDto value;
        private final Boolean currentNodeSegmentUncommon;

        public SelectionTreeNodeDto(String segmentationName, Map<String, SelectionTreeNodeDto> childrenBySegmentName,
                                    AccountsSpaceDto value, Boolean currentNodeSegmentUncommon) {
            this.childrenBySegmentName = childrenBySegmentName;
            this.segmentationName = segmentationName;
            this.value = value;
            this.currentNodeSegmentUncommon = currentNodeSegmentUncommon;
        }

        @Schema(description = "Localized segmentation's name.")
        public String getSegmentationName() {
            return segmentationName;
        }

        @Schema(description = "Children of internal node (map by segments names). For terminal node has null value.")
        public Map<String, SelectionTreeNodeDto> getChildrenBySegmentName() {
            return childrenBySegmentName;
        }

        @Schema(description = "Accounts space data for terminal node. For internal node has null value.")
        public AccountsSpaceDto getValue() {
            return value;
        }

        @Nullable
        @Schema(description = "Current node segment 'uncommon' flag. Null for root node.")
        public Boolean getCurrentNodeSegmentUncommon() {
            return currentNodeSegmentUncommon;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            SelectionTreeNodeDto that = (SelectionTreeNodeDto) o;
            return Objects.equals(childrenBySegmentName, that.childrenBySegmentName) &&
                    Objects.equals(segmentationName, that.segmentationName) &&
                    Objects.equals(value, that.value) &&
                    Objects.equals(currentNodeSegmentUncommon, that.currentNodeSegmentUncommon);
        }

        @Override
        public int hashCode() {
            return Objects.hash(childrenBySegmentName, value, segmentationName, currentNodeSegmentUncommon);
        }

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }
    }
}
