package ru.yandex.intranet.d.web.controllers.front

import io.swagger.v3.oas.annotations.Operation
import io.swagger.v3.oas.annotations.Parameter
import io.swagger.v3.oas.annotations.media.ArraySchema
import io.swagger.v3.oas.annotations.media.Content
import io.swagger.v3.oas.annotations.media.Schema
import io.swagger.v3.oas.annotations.responses.ApiResponse
import io.swagger.v3.oas.annotations.responses.ApiResponses
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.*
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.services.aggregates.QueryAggregatesService
import ru.yandex.intranet.d.util.response.Responses
import ru.yandex.intranet.d.web.model.resources.SelectionResourceTreeNodeDto
import ru.yandex.intranet.d.web.errors.Errors
import ru.yandex.intranet.d.web.model.ErrorCollectionDto
import ru.yandex.intranet.d.web.model.ProviderDto
import ru.yandex.intranet.d.web.model.aggregation.*
import ru.yandex.intranet.d.web.model.resources.ResourceSelectionListResponseDto
import ru.yandex.intranet.d.web.security.Auth
import ru.yandex.intranet.d.web.security.roles.UserRole
import java.security.Principal
import java.util.*


/**
 * Front aggregation API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserRole
@RestController
@RequestMapping("/front/aggregation")
open class FrontAggregationController(private val queryAggregatesService: QueryAggregatesService) {

    @Operation(summary = "Find subtree total.")
    @ApiResponses(ApiResponse(responseCode = "200", description = "Subtree total.", content = [Content(
        mediaType = MediaType.APPLICATION_JSON_VALUE,
        schema = Schema(implementation = FindSubtreeTotalResponseDto::class))]),
        ApiResponse(responseCode = "422", description = "'Validation failed' error response.", content = [Content(
            mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = ErrorCollectionDto::class))]))
    @PostMapping(value = ["/_findSubtreeTotal"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun findSubtreeTotal(
        @Parameter(description = "Find subtree total request.", required = true)
        @RequestBody request: FindSubtreeTotalRequestDto,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return queryAggregatesService.findSubtreeTotalMono(request, currentUser,
            locale).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(summary = "Rank subtree amounts.")
    @ApiResponses(ApiResponse(responseCode = "200", description = "Ranked subtree amounts.", content = [Content(
            mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = RankSubtreeAmountsResponseDto::class))]),
        ApiResponse(responseCode = "422", description = "'Validation failed' error response.", content = [Content(
            mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = ErrorCollectionDto::class))]))
    @PostMapping(value = ["/_rankSubtreeAmounts"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun rankSubtreeAmounts(
        @Parameter(description = "Rank subtree amounts request.", required = true)
        @RequestBody request: RankSubtreeAmountsRequestDto,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return queryAggregatesService.rankSubtreeAmountsMono(request, currentUser,
            locale).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(summary = "Find service totals.")
    @ApiResponses(ApiResponse(responseCode = "200", description = "Service totals.", content = [Content(
        mediaType = MediaType.APPLICATION_JSON_VALUE,
        schema = Schema(implementation = FindServiceTotalsResponseDto::class))]),
        ApiResponse(responseCode = "422", description = "'Validation failed' error response.", content = [Content(
            mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = ErrorCollectionDto::class))]))
    @PostMapping(value = ["/_findServiceTotals"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun findServiceTotals(
        @Parameter(description = "Find service totals request.", required = true)
        @RequestBody request: FindServiceTotalsRequestDto,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return queryAggregatesService.findServiceTotalsMono(request, currentUser,
            locale).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(
        description = "Get service providers",
        responses = [
            ApiResponse(
                responseCode = "200",
                description = "Get service providers",
                content = [Content(
                    mediaType = "application/json",
                    array = ArraySchema(schema = Schema(implementation = ProviderDto::class)),
                )]
            ),
            ApiResponse(
                responseCode = "422",
                description = "'Validation failed' error response.",
                content = [Content(
                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = Schema(implementation = ErrorCollectionDto::class))
                ]
            )
        ]
    )
    @GetMapping("/_providersByService")
    open fun findProvidersByService(
        @Parameter(description = "Root service id", required = true)
        @RequestParam rootServiceId: Long?,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        val request = GetProvidersByServiceRequestDto(rootServiceId)
        return queryAggregatesService.getProvidersByServiceMono(request, currentUser, locale)
            .map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(
        description = "Get resource selection tree",
        responses = [
            ApiResponse(
                responseCode = "200",
                description = "Get resource selection tree",
                content = [Content(
                    mediaType = "application/json",
                    array = ArraySchema(schema = Schema(
                        implementation = SelectionResourceTreeNodeDto::class)
                    ),
                )]
            ),
            ApiResponse(
                responseCode = "422",
                description = "'Validation failed' error response.",
                content = [Content(
                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = Schema(implementation = ErrorCollectionDto::class))
                ]
            )
        ]
    )
    @GetMapping("/_resourceSelectionTree")
    open fun getResourceSelectionTree(
        @Parameter(description = "Root service id", required = true)
        @RequestParam rootServiceId: Long?,
        @Parameter(description = "Provider id", required = true)
        @RequestParam providerId: String?,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        val request = GetResourceSelectionDto(rootServiceId, providerId)
        return queryAggregatesService.getResourceSelectionTreeMono(request, currentUser, locale)
            .map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(
        description = "Get resource selection list",
        responses = [
            ApiResponse(
                responseCode = "200",
                description = "Get resource selection list",
                content = [Content(
                    mediaType = "application/json",
                    array = ArraySchema(schema = Schema(
                        implementation = ResourceSelectionListResponseDto::class)
                    ),
                )]
            ),
            ApiResponse(
                responseCode = "422",
                description = "'Validation failed' error response.",
                content = [Content(
                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = Schema(implementation = ErrorCollectionDto::class))
                ]
            )
        ]
    )
    @GetMapping("/_resourceSelectionList")
    open fun getResourceSelectionList(
        @Parameter(description = "Root service id", required = true)
        @RequestParam rootServiceId: Long?,
        @Parameter(description = "Provider id", required = true)
        @RequestParam providerId: String?,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        val request = GetResourceSelectionDto(rootServiceId, providerId)
        return queryAggregatesService.getResourceSelectionListMono(request, currentUser, locale)
            .map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }
}
