package ru.yandex.intranet.d.web.controllers.front;

import java.security.Principal;
import java.util.Locale;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import org.springframework.http.ResponseEntity;
import org.springframework.lang.NonNull;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.services.folders.history.FolderHistoryService;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.SortOrderDto;
import ru.yandex.intranet.d.web.model.folders.front.history.FrontFolderOperationLogPageDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserRole;

/**
 * Front folders history controller
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
@UserRole
@RestController
@RequestMapping("/front/history/folders")
public class FrontFolderHistoryController {
    private final FolderHistoryService folderHistoryService;

    public FrontFolderHistoryController(FolderHistoryService folderHistoryService) {
        this.folderHistoryService = folderHistoryService;
    }

    @Operation(description = "Получить историю изменения для папки", responses = {
            @ApiResponse(responseCode = "200", description = "История изменения для папки",
                    content = @Content(mediaType = "application/json",
                            schema = @Schema(implementation = FrontFolderOperationLogPageDto.class))),
            @ApiResponse(responseCode = "404", description = "Папка не найдена",
                    content = @Content(mediaType = "application/json",
                            schema = @Schema(implementation = ErrorCollectionDto.class)))
    })
    @NonNull
    @GetMapping("/{folderId}")
    public Mono<ResponseEntity<?>> getFolderHistory(
            @Parameter(description = "Folder id.", required = true)
            @PathVariable String folderId,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            @Parameter(description = "Sort order.")
            @RequestParam(value = "sortOrder", required = false, defaultValue = "DESC") SortOrderDto sortOrderDto,
            Principal principal,
            Locale locale) {

        YaUserDetails currentUser = Auth.details(principal);

        final Mono<Result<FrontFolderOperationLogPageDto>> result = folderHistoryService
                .getFolderOperationLogsPage(folderId, new PageRequest(pageToken, limit), sortOrderDto, currentUser,
                        locale);

        return result.map(r -> r.match(Responses::okJson, Errors::toResponse));
    }

}
