package ru.yandex.intranet.d.web.controllers.front;

import java.security.Principal;
import java.util.Locale;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.lang.NonNull;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.services.folders.FolderService;
import ru.yandex.intranet.d.services.quotas.QuotasService;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.folders.FrontFolderDto;
import ru.yandex.intranet.d.web.model.folders.FrontFolderWithQuotesDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserRole;

/**
 * Front providers API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserRole
@RestController
@RequestMapping("/front/folders")
public class FrontFoldersController {

    private final FolderService folderService;
    private final QuotasService quotasService;

    public FrontFoldersController(FolderService folderService, QuotasService quotasService) {
        this.folderService = folderService;
        this.quotasService = quotasService;
    }

    @Operation(description = "Получить информацию о папках, провайдерах, ресурсах и квотах в сервисе",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Информация о папке",
                            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = FrontFolderWithQuotesDto.class)))})
    @NonNull
    @GetMapping
    @SuppressWarnings("ParameterNumber")
    public Mono<ResponseEntity<?>> getFolders(
            @Parameter(description = "Service id.", required = true)
            @RequestParam(value = "serviceId") long serviceId,
            @Parameter(description = "Provider id.", required = false)
            @RequestParam(value = "providerId", required = false) String providerId,
            @Parameter(description = "Resource id.", required = false)
            @RequestParam(value = "resourceId", required = false) String resourceId,
            @Parameter(description = "Continuation token.")
            @RequestParam(value = "continuationToken", required = false) String continuationToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal,
            Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);

        Mono<Result<FrontFolderWithQuotesDto>> pageForService = quotasService.getFrontPageForService(
                new PageRequest(continuationToken, limit),
                serviceId,
                providerId,
                resourceId,
                false,
                currentUser,
                locale);

        return pageForService.map(result -> result.match(
                page -> ResponseEntity.ok().contentType(MediaType.APPLICATION_JSON).body(page),
                Errors::toResponse
                )
        );
    }


    @Operation(description = "Получить информацию о папке", responses = {
            @ApiResponse(responseCode = "200", description = "Информация о папке",
                    content = @Content(mediaType = "application/json",
                            schema = @Schema(implementation = FrontFolderDto.class))),
            @ApiResponse(responseCode = "404", description = "Папка не найдена",
                    content = @Content(mediaType = "application/json",
                            schema = @Schema(implementation = ErrorCollectionDto.class)))
    })
    @NonNull
    @GetMapping("/{id}")
    public Mono<ResponseEntity<?>> getFolder(
            @Parameter(description = "Folder id.", required = true)
            @PathVariable String id,
            Principal principal,
            Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        return folderService.getFolder(id, currentUser, locale)
                .map(folderResult -> folderResult.match(
                        folder -> ResponseEntity.ok().contentType(MediaType.APPLICATION_JSON)
                                .body(FrontFolderDto.from(folder)),
                        Errors::toResponse
                ));
    }
}
