package ru.yandex.intranet.d.web.controllers.front

import io.swagger.v3.oas.annotations.Operation
import io.swagger.v3.oas.annotations.Parameter
import io.swagger.v3.oas.annotations.media.Content
import io.swagger.v3.oas.annotations.media.Schema
import io.swagger.v3.oas.annotations.responses.ApiResponse
import io.swagger.v3.oas.annotations.responses.ApiResponses
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.PathVariable
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RequestBody
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.RestController
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.kotlin.LoanId
import ru.yandex.intranet.d.services.loans.LoansService
import ru.yandex.intranet.d.util.response.Responses
import ru.yandex.intranet.d.web.errors.Errors
import ru.yandex.intranet.d.web.model.ErrorCollectionDto
import ru.yandex.intranet.d.web.model.loans.history.FrontGetLoansHistoryResponseDto
import ru.yandex.intranet.d.web.model.loans.front.FrontSearchLoansRequestDto
import ru.yandex.intranet.d.web.model.loans.front.FrontSearchLoansResponseDto
import ru.yandex.intranet.d.web.security.Auth
import ru.yandex.intranet.d.web.security.roles.UserRole
import java.security.Principal
import java.util.*

@UserRole
@RestController
@RequestMapping("/front/loans")
open class FrontLoansController(
    private val loansService: LoansService
) {
    @Operation(summary = "Find incoming/outcoming loans.")
    @ApiResponses(
        ApiResponse(responseCode = "200", description = "Loans.", content = [Content(
        mediaType = MediaType.APPLICATION_JSON_VALUE,
        schema = Schema(implementation = FrontSearchLoansResponseDto::class)
        )]),
        ApiResponse(responseCode = "422", description = "'Validation failed' error response.", content = [Content(
            mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = ErrorCollectionDto::class)
        )])
    )
    @PostMapping(value = ["_search"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun search(
        @RequestBody request: FrontSearchLoansRequestDto,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return loansService.searchLoansMono(request, currentUser, locale)
            .map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(summary = "Get one loan's history.")
    @ApiResponses(
        ApiResponse(responseCode = "200", description = "Loan's history events.", content = [Content(
            mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = FrontGetLoansHistoryResponseDto::class)
        )]),
        ApiResponse(responseCode = "422", description = "'Validation failed' error response.", content = [Content(
            mediaType = MediaType.APPLICATION_JSON_VALUE,
            schema = Schema(implementation = ErrorCollectionDto::class)
        )])
    )
    @GetMapping(value = ["/{loanId}/_history"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun getLoansHistory(
        @Parameter(description = "Loan ID", required = true)
        @PathVariable("loanId") loanId: LoanId,
        @Parameter(description = "Next page token", required = false)
        @RequestParam("pageToken") pageToken: String?,
        @Parameter(description = "Max amount of events in response", required = false)
        @RequestParam("limit") limit: Int?,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return loansService.getLoansHistoryMono(loanId, pageToken, limit, currentUser, locale)
            .map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }
}
