package ru.yandex.intranet.d.web.controllers.front;

import java.security.Principal;
import java.util.Locale;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.ArraySchema;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.lang.NonNull;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.services.folders.FolderService;
import ru.yandex.intranet.d.services.providers.ProvidersService;
import ru.yandex.intranet.d.services.quotas.QuotasService;
import ru.yandex.intranet.d.services.resources.ByAccountsFilterParameters;
import ru.yandex.intranet.d.services.resources.ExpandedResources;
import ru.yandex.intranet.d.services.resources.ResourceFilterParameters;
import ru.yandex.intranet.d.services.resources.ResourceTreeService;
import ru.yandex.intranet.d.services.resources.ResourcesService;
import ru.yandex.intranet.d.services.resources.SelectionResourceTreeNode;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.ProviderDto;
import ru.yandex.intranet.d.web.model.ProviderResourceDto;
import ru.yandex.intranet.d.web.model.folders.FrontFolderWithQuotesDto;
import ru.yandex.intranet.d.web.model.providers.ProviderUISettingsDto;
import ru.yandex.intranet.d.web.model.resources.SelectionResourceTreeNodeDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserRole;

import static ru.yandex.intranet.d.web.converters.SelectionResourceTreeNodeConverterKt.toSelectionResourceTreeNodeDto;
import static ru.yandex.intranet.d.web.util.ModelDtoConverter.providerDtoFromModel;

/**
 * Front API providers controller.
 *
 * @author Nikita Minin <spasitel@yandex-team.ru>
 */
@UserRole
@RestController
@RequestMapping("/front/providers")
public class FrontProvidersController {

    private final ProvidersService providersService;
    private final FolderService folderService;
    private final ResourcesService resourcesService;
    private final QuotasService quotasService;
    private final ResourceTreeService resourceTreeService;


    public FrontProvidersController(ProvidersService providersService,
                                    FolderService folderService,
                                    ResourcesService resourcesService,
                                    QuotasService quotasService, ResourceTreeService resourceTreeService) {
        this.providersService = providersService;
        this.folderService = folderService;
        this.resourcesService = resourcesService;
        this.quotasService = quotasService;
        this.resourceTreeService = resourceTreeService;
    }

    @Operation(description = "Получить информацию о провайдерах в сервисе",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Информация о провайдерах",
                            content = @Content(mediaType = "application/json",
                                    array = @ArraySchema(schema =
                                    @Schema(implementation = ProviderDto.class))))})
    @NonNull
    @GetMapping
    public Mono<ResponseEntity<?>> getProviders(
            @Parameter(description = "Flag to filter only providers with default quotas.")
            @RequestParam(value = "withDefaultQuotas", required = false, defaultValue = "false")
                    boolean withDefaultQuotas,
            Principal principal,
            Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        return providersService.getAllProviders(withDefaultQuotas, currentUser, locale)
                .map(r -> r.match(
                        providerModels -> Responses.okJson(providerModels.stream()
                                .map(entity -> providerDtoFromModel(entity, locale))
                                .collect(Collectors.toList())
                        ),
                        Errors::toResponse
                ));
    }

    @Operation(description = "Получить информацию о провайдере",
            responses = {@ApiResponse(responseCode = "200", description = "Информация о провайдере",
                    content = @Content(mediaType = "application/json",
                            schema = @Schema(implementation = ProviderDto.class)))})
    @NonNull
    @GetMapping("/{id}")
    public Mono<ResponseEntity<?>> getProvider(
            @Parameter(description = "Provider id.", required = true)
            @PathVariable String id,
            Principal principal,
            Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ProviderModel>> result = providersService.getById(id, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(providerDtoFromModel(entity, locale)),
                Errors::toResponse));
    }

    @Operation(description = "Получить информацию об ресурсах у провайдера в сервисе",
            responses = {@ApiResponse(responseCode = "200", description = "Информация о ресурсах",
                    content = @Content(mediaType = "application/json",
                            array = @ArraySchema(schema =
                            @Schema(implementation = ProviderResourceDto.class))))})
    @NonNull
    @GetMapping("/{id}/resources")
    public Mono<ResponseEntity<?>> getProviderResources(
            @Parameter(description = "Provider id.", required = true)
            @PathVariable String id,
            Principal principal,
            Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        return resourcesService.getAllProviderResources(currentUser, id, locale)
                .map(r -> r.match(rms -> Responses.okJson(rms.stream()
                        .map(rm -> toProviderResource(rm, locale))), Errors::toResponse));
    }

    @Operation(description = "Получить информацию о резервном фолдере провайдера и квотах в нём",
            responses = {@ApiResponse(responseCode = "200",
                    description = "Информация о резервном фолдере провайдера и квотах в нём",
                    content = @Content(mediaType = "application/json",
                            array = @ArraySchema(schema =
                            @Schema(implementation = FrontFolderWithQuotesDto.class))))})
    @NonNull
    @GetMapping("/{id}/_reservedQuotas")
    public Mono<ResponseEntity<?>> getProviderReservedQuotas(
            @Parameter(description = "Provider id.", required = true)
            @PathVariable String id,
            Principal principal,
            Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        return providersService.getById(id, currentUser, locale)
                .flatMap(r -> r.andThenMono(provider ->
                        folderService.getReservedFolderOfService(provider.getServiceId(),
                                currentUser,
                                locale
                        )).flatMap(k -> k.andThenMono(folder ->
                        quotasService.getProviderReservedQuotas(id, folder.getId(), currentUser, locale)
                )).map(result -> result.match(Responses::okJson, Errors::toResponse)));
    }

    private ProviderResourceDto toProviderResource(ResourceModel resource, Locale locale) {
        String description = Locales.select(resource.getDescriptionEn(), resource.getDescriptionRu(), locale);
        String name = Locales.select(resource.getNameEn(), resource.getNameRu(), locale);
        return new ProviderResourceDto(resource.getId(), resource.getProviderId(), description, name);
    }

    @Operation(summary = "Get resources selection tree.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested resources as selection tree.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = SelectionResourceTreeNodeDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "'Provider not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}/_resourceSelectionTree", produces = MediaType.APPLICATION_JSON_VALUE)
    @SuppressWarnings("checkstyle:ParameterNumber")
    public Mono<ResponseEntity<?>> getResourceSelectionTree(
            @Parameter(description = "Provider id", required = true)
            @PathVariable("id") String providerId,
            @Parameter(description = "Flag to add read-only resources to resource tree.")
            @RequestParam(value = "withReadOnly", required = false, defaultValue = "false")
                    boolean withReadOnly,
            @Parameter(description = "Flag to add unmanaged resources to resource tree.")
            @RequestParam(value = "withUnmanaged", required = false, defaultValue = "false")
                    boolean withUnmanaged,
            @Parameter(description = "Flag to add resources to resource tree from provider reserve account only .")
            @RequestParam(value = "providerReserveOnly", required = false, defaultValue = "false")
                    boolean providerReserveOnly,
            @Parameter(description = "Service id for resources filter by presence in service accounts.")
            @RequestParam(name = "serviceIdForFilter", required = false)
                    Long serviceIdForFilter,
            @Parameter(description = "Folder id for resources filter by presence in folder accounts.")
            @RequestParam(name = "folderIdForFilter", required = false)
                    String folderIdForFilter,
            @Parameter(description = "Account id for resources filter by presence in account.")
            @RequestParam(name = "accountIdForFilter", required = false)
                    String accountIdForFilter,
            @Parameter(description = "Flag for resource filter, take only resources where balance is positive.")
            @RequestParam(value = "balanceOnly", required = false, defaultValue = "false")
            boolean balanceOnly,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        ResourceFilterParameters filterParameters = ResourceFilterParameters.builder()
                .withReadOnly(withReadOnly)
                .withUnmanaged(withUnmanaged)
                .build();
        var byAccountsFilterParameters = new ByAccountsFilterParameters(
                serviceIdForFilter, folderIdForFilter, accountIdForFilter, providerReserveOnly, balanceOnly
        );
        Mono<Result<ExpandedResources<SelectionResourceTreeNode>>> result =
                resourceTreeService.getResourceSelectionTree(
                        providerId, filterParameters, byAccountsFilterParameters, currentUser, locale
                );
        return result.map(
                r -> r.match(t -> Responses.okJson(toSelectionResourceTreeNodeDto(t, locale)), Errors::toResponse));
    }

    @Operation(description = "Получить настройки UI провайдера по умолчанию. ",
            responses = {@ApiResponse(responseCode = "200",
                    description = "настройки UI провайдера по умолчанию",
                    content = @Content(mediaType = "application/json",
                            array = @ArraySchema(schema =
                            @Schema(implementation = ProviderUISettingsDto.class))))})
    @NonNull
    @GetMapping("/_defaultUISettings")
    public Mono<ResponseEntity<?>> getProviderDefaultUISettings(
            Principal principal,
            Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        return providersService.getDefaultUISettings(currentUser, locale)
                .map(result -> result.match(Responses::okJson, Errors::toResponse));
    }

}
