package ru.yandex.intranet.d.web.controllers.front;

import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.services.quotas.ProvisionDryRunService;
import ru.yandex.intranet.d.services.quotas.ProvisionLogicService;
import ru.yandex.intranet.d.services.quotas.ProvisionService;
import ru.yandex.intranet.d.services.quotas.QuotasService;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.FrontProvisionsDto;
import ru.yandex.intranet.d.web.model.folders.FrontFolderWithQuotesDto;
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionDryRunAnswerDto;
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionDryRunRequestDto;
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionsAnswerDto;
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionsRequestDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserRole;

/**
 * Front quotas API controller.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 * @since 7-12-2020
 */
@UserRole
@RestController
@RequestMapping("/front/quotas")
public class FrontQuotasController {
    private final ProvisionService provisionService;
    private final ProvisionDryRunService provisionDryRunService;
    private final QuotasService quotasService;
    private final ProvisionLogicService provisionLogicService;

    public FrontQuotasController(
            ProvisionService provisionService,
            QuotasService quotasService,
            ProvisionDryRunService provisionDryRunService,
            ProvisionLogicService provisionLogicService) {
        this.provisionService = provisionService;
        this.quotasService = quotasService;
        this.provisionDryRunService = provisionDryRunService;
        this.provisionLogicService = provisionLogicService;
    }

    @Operation(description = "Update provisions for account.", responses = {
            @ApiResponse(responseCode = "200", description = "Updated provisions.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = UpdateProvisionsAnswerDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "Not found.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))
    })
    @PostMapping(value = "/_updateProvisions", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> updateProvision(
            @Parameter(description = "Updated provisions.", required = true)
            @RequestBody UpdateProvisionsRequestDto updateProvisionsRequestDto,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<UpdateProvisionsAnswerDto>> result = provisionLogicService
                .updateProvisionMono(updateProvisionsRequestDto, currentUser, locale, false, idempotencyKey)
                .map(v -> v.apply(r -> r.getResult().orElseThrow()));
        return result.map(r -> r.match(Responses::okJson, Errors::toResponse));
    }

    @Operation(description = "Calculate new values for update provisions form with related resources.", responses = {
            @ApiResponse(responseCode = "200", description = "New values with related resources.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = UpdateProvisionDryRunAnswerDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404", description = "Not found.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))
    })
    @PostMapping(value = "/_updateProvisionDryRun", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> updateProvisionDryRun(
            @Parameter(description = "Updated provisions with related resources.", required = true)
            @RequestBody UpdateProvisionDryRunRequestDto request,
            Principal principal, Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        return provisionDryRunService.updateProvisionDryRun(request, currentUser, locale)
                .map(r -> r.match(Responses::okJson, Errors::toResponse));
    }

    @Operation(description = "Get resources quotas in folder", responses = {
            @ApiResponse(responseCode = "200", description = "Folders resources quotas",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = FrontFolderWithQuotesDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404",
                    description = "Some folders not found. See 'unknownFolderIds' key in error details.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))
    })
    @GetMapping(value = "/_folders", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getFolders(
            @Parameter(description = "Folders ids", required = true)
            @RequestParam("folderIds") List<String> folderIds,
            Principal principal,
            Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        return quotasService.getQuotasInFolders(Set.copyOf(folderIds), false, currentUser, locale)
                .map(result -> result.match(
                        dto -> ResponseEntity.ok().contentType(MediaType.APPLICATION_JSON).body(dto),
                        Errors::toResponse
                ));
    }

    @Operation(description = "Get all balances available for providing", responses = {
            @ApiResponse(responseCode = "200", description = "Balances available for providing",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = FrontProvisionsDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404",
                    description = "Some folders not found. See 'unknownFolderIds' key in error details.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))
    })
    @GetMapping(value = "/_getAllBalances", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getAllBalances(
            @Parameter(description = "Folders ids", required = true)
            @RequestParam("folderId") String folderId,
            @RequestParam("accountId") String accountId,
            Principal principal,
            Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        return quotasService.getAllBalances(folderId, accountId, currentUser, locale)
                .map(result -> result.match(
                        dto -> ResponseEntity.ok().contentType(MediaType.APPLICATION_JSON).body(dto),
                        Errors::toResponse
                ));
    }

    @Operation(description = "Get all non allocated quota", responses = {
            @ApiResponse(responseCode = "200", description = "Non allocated quotas",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = FrontProvisionsDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "404",
                    description = "Some folders not found. See 'unknownFolderIds' key in error details.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))
    })
    @GetMapping(value = "/_getAllNonAllocated", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getAllNonAllocated(
            @Parameter(description = "Folders ids", required = true)
            @RequestParam("folderId") String folderId,
            @RequestParam("accountId") String accountId,
            Principal principal,
            Locale locale
    ) {
        YaUserDetails currentUser = Auth.details(principal);
        return quotasService.getAllNonAllocated(folderId, accountId, currentUser, locale)
                .mapNotNull(result -> result.match(
                        dto -> ResponseEntity.ok().contentType(MediaType.APPLICATION_JSON).body(dto),
                        Errors::toResponse
                ));
    }
}
