package ru.yandex.intranet.d.web.controllers.front

import io.swagger.v3.oas.annotations.Operation
import io.swagger.v3.oas.annotations.media.Content
import io.swagger.v3.oas.annotations.media.Schema
import io.swagger.v3.oas.annotations.responses.ApiResponse
import io.swagger.v3.oas.annotations.responses.ApiResponses
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.PathVariable
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RestController
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.kotlin.ProviderId
import ru.yandex.intranet.d.services.resources.ResourcesExchangeService
import ru.yandex.intranet.d.util.response.Responses
import ru.yandex.intranet.d.web.errors.Errors
import ru.yandex.intranet.d.web.model.ErrorCollectionDto
import ru.yandex.intranet.d.web.model.providers.FrontGetProvidersForExchangeResponseDto
import ru.yandex.intranet.d.web.model.resources.exchangeable.FrontGetExchangeableResourcesResponseDto
import ru.yandex.intranet.d.web.security.Auth
import ru.yandex.intranet.d.web.security.roles.UserRole
import java.security.Principal
import java.util.*

@UserRole
@RestController
@RequestMapping("/front/exchange")
open class FrontResourceExchangeController(
    private val resourcesExchangeService: ResourcesExchangeService
) {
    @Operation(summary = "Get exchangeable resource types and segments for provider.")
    @ApiResponses(
        ApiResponse(
            responseCode = "200", description = "Exchangeable types and segments.", content = [Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE,
                schema = Schema(implementation = FrontGetExchangeableResourcesResponseDto::class)
            )]
        ),
        ApiResponse(
            responseCode = "422", description = "'Validation failed' error response.", content = [Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE,
                schema = Schema(implementation = ErrorCollectionDto::class)
            )]
        )
    )
    @GetMapping("/{providerId}/_exchangeableResourceTypes")
    open fun getExchangeableResourceTypesByProvider(
        @PathVariable providerId: ProviderId?,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return resourcesExchangeService.getExchangeableResourceTypesByProviderMono(providerId, currentUser, locale)
            .map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    @Operation(summary = "Get list of providers that support resources exchange.")
    @ApiResponses(
        ApiResponse(
            responseCode = "200", description = "Providers list for resources exchange.", content = [Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE,
                schema = Schema(implementation = FrontGetProvidersForExchangeResponseDto::class)
            )]
        )
    )
    @GetMapping("/_providers")
    open fun getProvidersForResourcesExchange(
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return resourcesExchangeService.getProvidersForResourcesExchangeMono(currentUser, locale)
            .map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }
}
