package ru.yandex.intranet.d.web.controllers.front;

import java.security.Principal;
import java.util.Locale;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import org.springframework.http.ResponseEntity;
import org.springframework.lang.NonNull;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.services.resources.ExpandedResources;
import ru.yandex.intranet.d.services.resources.ResourcesService;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.ResourceDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserRole;

/**
 * Front resources API controller.
 *
 * @author Nikita Minin <spasitel@yandex-team.ru>
 */
@UserRole
@RestController
public class FrontResourcesController {

    private final ResourcesService resourcesService;

    public FrontResourcesController(ResourcesService resourcesService) {
        this.resourcesService = resourcesService;
    }

    @Operation(description = "Получить информацию о ресурсе",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Информация о ресурсе",
                            content = @Content(mediaType = "application/json",
                                    schema = @Schema(implementation = ResourceDto.class))),
                    @ApiResponse(responseCode = "404", description = "Ресурс не найден",
                            content = @Content(mediaType = "application/json",
                                    schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @NonNull
    @GetMapping("front/resources/{id}")
    public Mono<ResponseEntity<?>> getResource(
            @Parameter(description = "Resource id.", required = true)
            @PathVariable String id,
            Principal principal,
            Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedResources<ResourceModel>>> result = resourcesService
                .getByIdFront(id, currentUser, locale);
        return result.map(r -> r.match(resource -> Responses.okJson(new ResourceDto(
                resource.getResources(), locale, resource.getUnitsEnsembles(), resource.getResourceSegmentations(),
                resource.getResourceSegments()
        )), Errors::toResponse));
    }
}
