package ru.yandex.intranet.d.web.controllers.front;

import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.services.folders.FolderService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.folders.FrontFolderDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * Front service controller
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/front/services")
public class FrontServiceController {

    private final FolderService folderService;

    public FrontServiceController(FolderService folderService) {
        this.folderService = folderService;
    }

    @Operation(summary = "Get one folders page for the ABC service.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested folders page for the ABC service.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FolderPageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{serviceId}/folders", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getFoldersPage(
            @Parameter(description = "ABC service id", required = true)
            @PathVariable Long serviceId,
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            @Parameter(description = "'With deleted' flag.")
            @RequestParam(value = "withDeleted", required = false, defaultValue = "false") boolean withDeleted,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Page<FolderModel>>> result = folderService
                .listFoldersByService(serviceId, withDeleted, new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p)), Errors::toResponse));
    }

    private PageDto<FrontFolderDto> toPage(Page<FolderModel> p) {
        return new PageDto<>(p.getItems().stream().map(FrontFolderDto::from)
                .collect(Collectors.toList()), p.getContinuationToken().orElse(null));
    }

    @Schema(description = "Folders page.")
    private static final class FolderPageDto extends PageDto<FrontFolderDto> {
        private FolderPageDto(List<FrontFolderDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }
}
