package ru.yandex.intranet.d.web.controllers.front;

import java.security.Principal;
import java.util.Locale;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.services.transfer.DryRunTransferRequestService;
import ru.yandex.intranet.d.services.transfer.TransferRequestAnswerService;
import ru.yandex.intranet.d.services.transfer.TransferRequestLogicService;
import ru.yandex.intranet.d.services.transfer.TransferRequestService;
import ru.yandex.intranet.d.services.transfer.model.ExpandedTransferRequests;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestsSearchParameters;
import ru.yandex.intranet.d.services.transfer.model.dryrun.DryRunTransferRequestResult;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontPutTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontSingleTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestHistoryEventsPageDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestSearchDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestVotingDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestsPageDto;
import ru.yandex.intranet.d.web.model.transfers.front.dryrun.FrontDryRunCreateTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.front.dryrun.FrontDryRunSingleTransferResultDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserRole;

/**
 * Transfer requests frontend API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserRole
@RestController
@RequestMapping("/front/transfers")
public class FrontTransferRequestsController {

    private final TransferRequestService transferRequestService;
    private final TransferRequestLogicService transferRequestLogicService;
    private final TransferRequestAnswerService answerService;
    private final DryRunTransferRequestService dryRunRequestService;

    public FrontTransferRequestsController(TransferRequestService transferRequestService,
                                           TransferRequestLogicService transferRequestLogicService,
                                           TransferRequestAnswerService answerService,
                                           DryRunTransferRequestService dryRunRequestService) {
        this.transferRequestService = transferRequestService;
        this.transferRequestLogicService = transferRequestLogicService;
        this.answerService = answerService;
        this.dryRunRequestService = dryRunRequestService;
    }

    @Operation(summary = "Get one transfer request by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested transfer request.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FrontSingleTransferRequestDto.class))),
            @ApiResponse(responseCode = "404", description = "'Transfer request not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Transfer request id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedTransferRequests<TransferRequestModel>>> result = transferRequestService
                .getOne(id, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(answerService
                        .toSingleRequest(entity, currentUser, locale)), Errors::toResponse));
    }

    @Operation(summary = "Create new transfer request.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Created transfer request.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FrontSingleTransferRequestDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> create(
            @Parameter(description = "Transfer request to create", required = true)
            @RequestBody FrontCreateTransferRequestDto transferRequest,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey,
            @RequestParam(value = "delayValidation", required = false, defaultValue = "false") boolean delayValidation,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedTransferRequests<TransferRequestModel>>> result = transferRequestLogicService
                .createMono(transferRequest, currentUser, locale, false, idempotencyKey, delayValidation);
        return result.map(r -> r.match(entity -> Responses.okJson(answerService
                        .toSingleRequest(entity, currentUser, locale)), Errors::toResponse));
    }

    @Operation(summary = "Update existing transfer request.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated transfer request.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FrontSingleTransferRequestDto.class))),
            @ApiResponse(responseCode = "404", description = "'Transfer request not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "412", description = "'Version mismatch' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PutMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> put(
            @Parameter(description = "Transfer request id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Current transfer request version", required = true)
            @RequestParam("version") Long version,
            @Parameter(description = "Updated transfer request", required = true)
            @RequestBody FrontPutTransferRequestDto transferRequest,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey,
            @RequestParam(value = "delayValidation", required = false, defaultValue = "false") boolean delayValidation,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedTransferRequests<TransferRequestModel>>> result = transferRequestLogicService
                .putMono(id, version, transferRequest, currentUser, locale, false, idempotencyKey,
                        delayValidation);
        return result.map(r -> r.match(entity -> Responses.okJson(answerService
                        .toSingleRequest(entity, currentUser, locale)), Errors::toResponse));
    }

    @Operation(summary = "Cancel existing transfer request.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Cancelled transfer request.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FrontSingleTransferRequestDto.class))),
            @ApiResponse(responseCode = "404", description = "'Transfer request not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/{id}/_cancel", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> cancel(
            @Parameter(description = "Transfer request id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Current transfer request version", required = true)
            @RequestParam("version") Long version,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedTransferRequests<TransferRequestModel>>> result = transferRequestLogicService
                .cancelMono(id, version, currentUser, locale, idempotencyKey);
        return result.map(r -> r.match(entity -> Responses.okJson(answerService
                        .toSingleRequest(entity, currentUser, locale)), Errors::toResponse));
    }

    @Operation(summary = "Vote for existing transfer request.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Updated transfer request.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FrontSingleTransferRequestDto.class))),
            @ApiResponse(responseCode = "404", description = "'Transfer request not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/{id}/_vote", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> vote(
            @Parameter(description = "Transfer request id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Current transfer request version", required = true)
            @RequestParam("version") Long version,
            @Parameter(description = "Vote parameters", required = true)
            @RequestBody FrontTransferRequestVotingDto voteParameters,
            @RequestHeader(name = "Idempotency-Key", required = false) String idempotencyKey,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedTransferRequests<TransferRequestModel>>> result = transferRequestLogicService
                .voteMono(id, version, voteParameters, currentUser, locale, idempotencyKey);
        return result.map(r -> r.match(entity -> Responses.okJson(answerService
                        .toSingleRequest(entity, currentUser, locale)), Errors::toResponse));
    }

    @Operation(summary = "Search transfer requests.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Transfer requests page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FrontTransferRequestsPageDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/_search", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> search(
            @Parameter(description = "Search parameters", required = true)
            @RequestBody FrontTransferRequestSearchDto searchParameters,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedTransferRequests<Page<TransferRequestModel>>>> result = transferRequestService
                .search(new TransferRequestsSearchParameters(searchParameters), new PageRequest(searchParameters
                        .getPageToken().orElse(null), searchParameters.getLimit().orElse(null)),
                        currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(answerService
                        .toRequestsPage(entity, currentUser, locale)), Errors::toResponse));
    }

    @Operation(summary = "Get history for one transfer request.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Transfer requests history page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FrontTransferRequestHistoryEventsPageDto.class))),
            @ApiResponse(responseCode = "404", description = "'Transfer request not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}/history", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getHistoryByRequest(
            @Parameter(description = "Transfer request id", required = true)
            @PathVariable("id") String id,
            @Parameter(description = "Page token")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit for response items count")
            @RequestParam(value = "limit", required = false) Long limit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedTransferRequests<Page<TransferRequestHistoryModel>>>> result = transferRequestService
                .history(id, new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(answerService.toHistoryPage(entity, locale)),
                Errors::toResponse));
    }

    @Operation(summary = "Dry run new transfer request.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Dry run result.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FrontDryRunSingleTransferResultDto.class))),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "_dryRun", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> dryRun(
            @Parameter(description = "Transfer request to create", required = true)
            @RequestBody FrontDryRunCreateTransferRequestDto transferRequest,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<ExpandedTransferRequests<DryRunTransferRequestResult>>> result = dryRunRequestService
                .dryRun(transferRequest, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(answerService
                .toSingleDryRunResult(entity, currentUser, locale)), Errors::toResponse));
    }
}
