package ru.yandex.intranet.d.web.controllers.front;

import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.units.UnitsEnsemblesService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.PageDto;
import ru.yandex.intranet.d.web.model.units.front.FrontUnitsEnsembleDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserOrServiceRole;

/**
 * Units ensembles public API controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserOrServiceRole
@RestController
@RequestMapping("/front/unitsEnsembles")
public class FrontUnitsEnsemblesController {

    private final UnitsEnsemblesService unitsEnsemblesService;

    public FrontUnitsEnsemblesController(UnitsEnsemblesService unitsEnsemblesService) {
        this.unitsEnsemblesService = unitsEnsemblesService;
    }

    @Operation(summary = "Get one units ensemble by id.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested units ensemble.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FrontUnitsEnsembleDto.class))),
            @ApiResponse(responseCode = "404", description = "'Units ensemble not found' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getOne(
            @Parameter(description = "Units ensemble id", required = true)
            @PathVariable("id") String id,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<UnitsEnsembleModel>> result = unitsEnsemblesService.getById(id, currentUser, locale);
        return result.map(r -> r.match(entity -> Responses.okJson(new FrontUnitsEnsembleDto(entity, locale)),
                Errors::toResponse));
    }

    @Operation(summary = "Get one units ensembles page.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested units ensembles page.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = FrontUnitsEnsemblePageDto.class))),
            @ApiResponse(responseCode = "400", description = "'Bad request' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @GetMapping(value = "", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getPage(
            @Parameter(description = "Page token.")
            @RequestParam(value = "pageToken", required = false) String pageToken,
            @Parameter(description = "Limit.")
            @RequestParam(value = "limit", required = false, defaultValue = "100") Long limit,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Page<UnitsEnsembleModel>>> result = unitsEnsemblesService
                .getPage(new PageRequest(pageToken, limit), currentUser, locale);
        return result.map(r -> r.match(p -> Responses.okJson(toPage(p, locale)), Errors::toResponse));
    }

    private FrontUnitsEnsemblePageDto toPage(Page<UnitsEnsembleModel> p, Locale locale) {
        return new FrontUnitsEnsemblePageDto(p.getItems().stream().map(m -> new FrontUnitsEnsembleDto(m, locale))
                .collect(Collectors.toList()), p.getContinuationToken().orElse(null));
    }

    @Schema(description = "Units ensembles page.")
    private static final class FrontUnitsEnsemblePageDto extends PageDto<FrontUnitsEnsembleDto> {
        private FrontUnitsEnsemblePageDto(List<FrontUnitsEnsembleDto> items, String continuationToken) {
            super(items, continuationToken);
        }
    }

}
