package ru.yandex.intranet.d.web.controllers.idm;

import java.util.Map;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.services.idm.IdmService;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.web.model.idm.IdmResult;
import ru.yandex.intranet.d.web.model.idm.IdmUpdate;
import ru.yandex.intranet.d.web.model.idm.IdmUpdateDto;
import ru.yandex.intranet.d.web.security.roles.IdmServiceRole;

/**
 * IDM API controller.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 * @since 06.10.2020
 */
@RestController
@RequestMapping(path = "/idm", produces = MediaType.APPLICATION_JSON_VALUE)
@IdmServiceRole
public class IdmController {

    private static final Logger LOG = LoggerFactory.getLogger(IdmController.class);

    private final IdmService idmService;
    private final ObjectReader roleReader;

    public IdmController(IdmService idmService, @Qualifier("idmJsonObjectMapper") ObjectMapperHolder objectMapper) {
        ObjectMapper mapper = objectMapper.getObjectMapper();
        TypeReference<Map<String, String>> roleTypeReference = new TypeReference<>() {
        };
        this.idmService = idmService;
        this.roleReader = mapper.readerFor(roleTypeReference);
    }

    @Operation(summary = "Get info about IDM systems and available roles.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Info about IDM systems and available roles.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE))})
    @GetMapping(path = "/info")
    public Mono<ResponseEntity<IdmResult>> getInfo() {
        return idmService.getRolesInfo()
                .map(Responses::okJson)
                .doOnError(e -> LOG.error("Failed while get idm roles info", e))
                .onErrorResume(throwable -> Mono.just(Responses.okJson(IdmResult.fromThrowable(throwable))));
    }

    @Operation(summary = "Get all users with roles.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "All users with roles.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE))})
    @GetMapping(path = "/get-all-roles")
    public Mono<ResponseEntity<IdmResult>> getAllRoles() {
        return idmService.getAllRoles()
                .map(Responses::okJson)
                .doOnError(e -> LOG.error("Failed while get all users roles to user", e))
                .onErrorResume(throwable -> Mono.just(Responses.okJson(IdmResult.fromThrowable(throwable))));
    }

    @Operation(summary = "Add role to user.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Status of adding new role.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE))})
    @PostMapping(value = "/add-role", consumes = {MediaType.APPLICATION_FORM_URLENCODED_VALUE})
    public Mono<ResponseEntity<IdmResult>> addRole(
            @Parameter(description = "User uid and role json to add", required = true) IdmUpdateDto idmUpdateDto
    ) {
        return idmService.addRole(idmUpdateFromIdmUpdateDto(idmUpdateDto))
                .map(Responses::okJson)
                .doOnError(e -> LOG.error("Failed while add roles to user", e))
                .onErrorResume(throwable -> Mono.just(Responses.okJson(IdmResult.fromThrowable(throwable))));
    }

    @Operation(summary = "Remove role from user.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Status of removing role from user.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE))})
    @PostMapping(value = "/remove-role", consumes = {MediaType.APPLICATION_FORM_URLENCODED_VALUE})
    public Mono<ResponseEntity<IdmResult>> removeRole(
            @Parameter(description = "User uid and role json to add", required = true) IdmUpdateDto idmUpdateDto
    ) {
        return idmService.removeRole(idmUpdateFromIdmUpdateDto(idmUpdateDto))
                .map(Responses::okJson)
                .doOnError(e -> LOG.error("Failed while removed roles from user", e))
                .onErrorResume(throwable -> Mono.just(Responses.okJson(IdmResult.fromThrowable(throwable))));
    }

    private IdmUpdate idmUpdateFromIdmUpdateDto(IdmUpdateDto updateDto) {
        return IdmUpdate.builder()
                .uid(updateDto.getUid())
                .roleMap(readRoleValue(updateDto.getRole()))
                .build();
    }

    private Map<String, String> readRoleValue(String role) {
        try {
            return roleReader.readValue(role);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Wrong role json.");
        }
    }
}
