package ru.yandex.intranet.d.web.controllers.legacy

import io.swagger.v3.oas.annotations.Operation
import io.swagger.v3.oas.annotations.Parameter
import io.swagger.v3.oas.annotations.media.Content
import io.swagger.v3.oas.annotations.media.Schema
import io.swagger.v3.oas.annotations.responses.ApiResponse
import org.springframework.http.HttpStatus
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.*
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.services.legacy.DispenserApiCompatibleProjectService
import ru.yandex.intranet.d.services.legacy.DispenserApiCompatibleProvisionService
import ru.yandex.intranet.d.services.legacy.DispenserApiCompatibleProvisionService.TransferQuotaFromParentParams
import ru.yandex.intranet.d.util.response.Responses
import ru.yandex.intranet.d.web.controllers.legacy.dto.DiQuota
import ru.yandex.intranet.d.web.controllers.legacy.dto.UpdateQuotaRequestBody
import ru.yandex.intranet.d.web.errors.Errors
import ru.yandex.intranet.d.web.model.ErrorCollectionDto
import ru.yandex.intranet.d.web.model.legacy.DispenserGetProjectsListResponseDto
import ru.yandex.intranet.d.web.model.legacy.DispenserGetQuotasResponseDto
import ru.yandex.intranet.d.web.security.Auth
import ru.yandex.intranet.d.web.security.roles.HardwareOrderServiceRole
import java.security.Principal
import java.util.*

/**
 * Dispenser API compatible controller.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 26-04-2022
 */
@RestController
@HardwareOrderServiceRole
@RequestMapping("/dispenser")
open class DispenserApiCompatibleController(
    private val dispenserApiCompatibleProjectService: DispenserApiCompatibleProjectService,
    private val dispenserApiCompatibleProvisionService: DispenserApiCompatibleProvisionService
) {
    // GET /db/api/v2/quotas, передаем в квери project
    @Operation(
        description = "Get project quotas",
        responses = [
            ApiResponse(
                responseCode = "200",
                description = "Project quotas",
                content = [Content(
                    mediaType = "application/json",
                    schema = Schema(implementation = DispenserGetQuotasResponseDto::class)
                )]
            ),
            ApiResponse(
                responseCode = "404",
                description = "Service, provider or default folder not found",
                content = [Content(
                    mediaType = "application/json",
                    schema = Schema(implementation = ErrorCollectionDto::class)
                )]
            )]
    )
    @GetMapping("/db/api/v2/quotas")
    open fun getQuotas(
        @Parameter(description = "Service (project) key", required = true)
        @RequestParam(required = true) project: String,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        val currentUser = Auth.details(principal)
        return dispenserApiCompatibleProvisionService.getMdbProvisionByServiceSlugMono(
            serviceSlug = project, currentUser, locale
        ).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    // GET /common/api/v1/projects, передаем в квери project - слаг абц сервиса с квотой
    @Operation(
        description = "Get services with child",
        responses = [
            ApiResponse(
                responseCode = "200",
                description = "Services with child",
                content = [Content(
                    mediaType = "application/json",
                    schema = Schema(implementation = DispenserGetProjectsListResponseDto::class)
                )]
            )]
    )
    @GetMapping(value = ["/common/api/v1/projects"], produces = [MediaType.APPLICATION_JSON_VALUE])
    open fun getProjects(
        @Parameter(description = "Service slug", required = true)
        @RequestParam(name = "project") projects: List<String>,
        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        if (projects.isEmpty()) {
            return Mono.just(ResponseEntity
                .status(HttpStatus.BAD_REQUEST)
                .body("Parameter 'project' is required.")
            )
        }
        return dispenserApiCompatibleProjectService.getServicesBySlugMono(projects, locale)
            .map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }

    // POST /db/api/v1/quotas/{project_key}/{service_key}/{resource_key}/{quota_spec_key}
    @Operation(
        description = "Update project quotas",
        responses = [
            ApiResponse(
                responseCode = "200",
                description = "Project new quotas",
                content = [Content(
                    mediaType = "application/json",
                    schema = Schema(implementation = DiQuota::class)
                )]
            ),
            ApiResponse(
                responseCode = "404",
                description = "Service, provider or resource not found",
                content = [Content(
                    mediaType = "application/json",
                    schema = Schema(implementation = ErrorCollectionDto::class)
                )]
            )]
    )
    @PostMapping("/db/api/v1/quotas/{project_key}/{service_key}/{resource_key}/{quota_spec_key}")
    open fun update(
        @Parameter(description = "ABC service slag", required = true)
        @PathVariable("project_key") projectKey: String,

        @Parameter(description = "Provider key", required = true)
        @PathVariable("service_key") serviceKey: String,

        @Parameter(description = "Resource key", required = true)
        @PathVariable("resource_key") resourceKey: String,

        @Parameter(description = "Extended resource key", required = true)
        @PathVariable("quota_spec_key") quotaSpecKey: String,

        @Parameter(description = "Update quota", required = true)
        @RequestBody maxValue: UpdateQuotaRequestBody,

        @RequestHeader("X-Yandex-UID", required = true)
        uid: String,

        principal: Principal,
        locale: Locale
    ): Mono<ResponseEntity<*>> {
        return dispenserApiCompatibleProvisionService.transferQuotaFromParentMono(
            TransferQuotaFromParentParams(
                projectKey,
                serviceKey,
                resourceKey,
                quotaSpecKey,
                maxValue.maxValue,
                maxValue.ownMaxValue,
                maxValue.unit,
                maxValue.segments,
            ),
            uid, locale
        ).map { it.match({ r -> Responses.okJson(r) }, Errors::toResponse) }
    }
}
