package ru.yandex.intranet.d.web.controllers.recipe;

import java.security.Principal;
import java.util.Locale;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Profile;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.services.folders.FolderService;
import ru.yandex.intranet.d.services.recipe.RecipeService;
import ru.yandex.intranet.d.util.response.Responses;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.recipe.AccountsDto;
import ru.yandex.intranet.d.web.model.recipe.AccountsQuotasDto;
import ru.yandex.intranet.d.web.model.recipe.CreateServiceDto;
import ru.yandex.intranet.d.web.model.recipe.CreateUserDto;
import ru.yandex.intranet.d.web.security.Auth;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.security.roles.UserRole;

/**
 * Recipe controller.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@UserRole
@RestController
@RequestMapping("/recipe")
@Profile({"test-recipe"})
public class RecipeController {

    private final MessageSource messages;
    private final RecipeService recipeService;
    private final FolderService folderService;

    public RecipeController(@Qualifier("messageSource") MessageSource messages,
                            RecipeService recipeService,
                            FolderService folderService) {
        this.messages = messages;
        this.recipeService = recipeService;
        this.folderService = folderService;
    }

    @Operation(summary = "Create new service.")
    @ApiResponses({@ApiResponse(responseCode = "204", description = "Service was successfully created."),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/service", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> createService(
            @Parameter(description = "Service to create", required = true)
            @RequestBody CreateServiceDto service,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Void>> result = recipeService.createService(service, currentUser, locale);
        return result.map(r -> r.match(entity -> ResponseEntity.noContent().build(), Errors::toResponse));
    }

    @Operation(summary = "Create new user.")
    @ApiResponses({@ApiResponse(responseCode = "204", description = "User was successfully created."),
            @ApiResponse(responseCode = "422", description = "'Validation failed' error response.",
                    content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorCollectionDto.class)))})
    @PostMapping(value = "/user", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> createUser(
            @Parameter(description = "User to create", required = true)
            @RequestBody CreateUserDto user,
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<Void>> result = recipeService.createUser(user, currentUser, locale);
        return result.map(r -> r.match(entity -> ResponseEntity.noContent().build(), Errors::toResponse));
    }

    @Operation(summary = "Create default folders.")
    @ApiResponses({@ApiResponse(responseCode = "204", description = "Default folders were successfully created.")})
    @PostMapping(value = "/_createDefaultFolders", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> createDefaultFolders(
            Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        if (currentUser.getUser().isEmpty() || !currentUser.getUser().get().getDAdmin()) {
            return Mono.just(Errors.toResponse(ErrorCollection.builder().addError(TypedError
                    .forbidden(messages.getMessage("errors.access.denied", null, locale))).build()));
        }
        Mono<Long> result = folderService.addDefaultFolders();
        return result.map(r -> ResponseEntity.noContent().build());
    }

    @Operation(summary = "Get accounts.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested accounts.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = AccountsDto.class)))})
    @GetMapping(value = "/accounts", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getAccounts(Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<AccountsDto>> result = recipeService.getAccounts(currentUser, locale);
        return result.map(r -> r.match(Responses::okJson, Errors::toResponse));
    }

    @Operation(summary = "Get accounts quotas.")
    @ApiResponses({@ApiResponse(responseCode = "200", description = "Requested accounts quotas.",
            content = @Content(mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = AccountsQuotasDto.class)))})
    @GetMapping(value = "/accountsQuotas", produces = MediaType.APPLICATION_JSON_VALUE)
    public Mono<ResponseEntity<?>> getAccountsQuotas(Principal principal, Locale locale) {
        YaUserDetails currentUser = Auth.details(principal);
        Mono<Result<AccountsQuotasDto>> result = recipeService.getAccountsQuotas(currentUser, locale);
        return result.map(r -> r.match(Responses::okJson, Errors::toResponse));
    }

}
